""" Python XML Templating Language

    Modules in this package:

    reference   - reference implementation
    optimised   - optimised implementation
    constants   - constants shared across all modules
    exceptions  - exceptions that may be raised by an implementation
    serialisers - write DOM trees in various formats
    utilities   - general-purpose helper functions
    pxdom       - stand-alone pure-Python DOM Level 3 implementation
    fixdom      - bug-fix wrappers to other DOM implementations

    Templates in this package:

    __error__   - PXTL template to pretty-print a PXTL error

    Functions provided at pxtl module level:

    processFile - forwarded to pxtl.reference or pxtl.optimised
    encodeText  - forwarded to pxtl.utilties
    decodeName  - forwarded to pxtl.utilties

    Example usage:

    import pxtl
    g= { 'itemsToShow': 3 }
    pxtl.processFile(os.path.join(templateDir, 'items.px'), globals= g)

    Other functions may be available directly from the implementation module,
    eg. pxtl.reference.
"""

__version__= 1,5
__author__ = 'Andrew Clover <and@doxdesk.com>'

__all__= [
  'reference', 'optimised',
  'constants', 'exceptions',
  'serialisers', 'utilities',
  'pxdom', 'fixdom',
  'processFile', 'encodeText', 'decodeName'
]
__data__= ['__error__.px']

import sys
from pxtl.constants import *
from pxtl.utilities import encodeText, decodeName


def processFile(
  path, writer= None, globals= None, debug= False, headers= False,
  dom= None, bytecodebase= {}
):
  """ Use the most appropriate PXTL implementation available to load a file,
      transform it, and send the serialised output to a stream. Optional
      arguments:

      writer  - stream-like object to send serialised output (default stdout)
      globals - dictionary to use as template global scope (default {})
      debug   - output a debugging page if there's an error (default False)
      headers - output HTTP response headers, for CGI scripts (default False)
      dom     - DOM L3 Core/XML/Load implementation to use (default pxdom)

      bytecodebase - mapping of source directory paths to bytecode (.pxc)
        cache directory paths, used by the optimised implementation only
        (default {}). If no key in the mapping matches an ancestor of a source
        template being compiled, the .pxc is dropped in the same directory as
        the source. Pass None to not use bytecode.
  """
  useOptimised= bytecodebase is not None

  if useOptimised:
    import pxtl.optimised
    module= pxtl.optimised
  else:
    import pxtl.reference
    module= pxtl.reference

  module.processFile(path, writer, globals, debug, headers, dom, bytecodebase)


__usage__= 'Usage: __init__.py [-d] template.px'

if __name__=='__main__':
  if len(sys.argv)<2:
    sys.stderr.write(__usage__+'\n')
    sys.exit(-1)
  processFile(sys.argv[-1], debug= ('-d' in sys.argv[1:-1]), headers= True)
