""" Exceptions generated by either PXTL implementation.
"""

import string, types, traceback
from pxtl.constants import *
class NullNode: pass


class PXTLError(Exception):
  """ Base class for all errors caused by bad code in templates. Parameter
      'context' is a DOM Node indicating the bit of the template that caused the
      error.
  """
  def __init__(self, context):
    Exception.__init__(self)
    self.context= context
  def __str__(self):
    try:
      desc= '%s, at: %s' % (self.describeError(), self.describeContext())
      if type(desc)==UnicodeType:
        return string.join(map(lambda c: (c, '?')[ord(c)>=128], desc), '')
      return desc
    except:
      traceback.print_exc()
      raise
  def describeError(self):
    return 'PXTL error'
  def describeContext(self):
    if self.context is None:
      return '(unknown)'
    node= self.context

    if node.nodeType==node.ELEMENT_NODE:
      desc= '<%s>' % node.nodeName
      node= node.parentNode
    elif node.nodeType==node.ATTRIBUTE_NODE:
      desc= 'Attribute "%s"' % node.nodeName
      node= node.ownerElement
    elif node.nodeType==node.PROCESSING_INSTRUCTION_NODE:
      desc= 'PI "%s"' % node.nodeName
      node= node.parentNode
    else:
      desc= node.nodeName
    while node is not None and node.nodeType==node.ELEMENT_NODE:
      desc= '%s in <%s>' % (desc, node.nodeName)
      node= node.parentNode
    try:
      location= self.context.pxdomLocation
      if location.uri is not None:
        desc= desc+', '+location.uri
      if location.lineNumber!=-1:
        desc= desc+', line '+str(location.lineNumber)
      if location.columnNumber!=-1:
        desc= desc+' char '+str(location.columnNumber)
    except AttributeError:
      pass
    return desc


# Generic exceptions
#
class PXTLSyntaxError(PXTLError):
  """ Template is not valid PXTL.
  """
  def describeError(self):
    return "Invalid PXTL syntax"

class PXTLNameError(PXTLError):
  """ Template includes item with unknown name.
  """
  def __init__(self, context, name):
    PXTLError.__init__(self, context)
    self.name= name
  def describeError(self):
    return "unknown name %s" % self.name

class PXTLTypeError(PXTLError):
  """ A PXTL attribute evaluated to the wrong type. Parameter 'given' is what
      was evaluated, 'needed' is a type object specifying what was expected.
  """
  def __init__(self, context, given, needed):
    PXTLError.__init__(self, context)
    self.given= given
    self.needed= needed
  def describeError(self):
    return '%s required, got %s (%s)' % tuple(map(str,
      (self.needed, type(self.given), self.given)
    ))

class PXTLValueError(PXTLError):
  """ A PXTL attribute evaluated to an unknown value. Parameter 'given' is what
      was evaluated, 'needed' is a list of possible values.
  """
  def __init__(self, context, given, needed):
    PXTLError.__init__(self, context)
    self.given= given
    self.needed= needed
  def describeError(self):
    return '%s or %s required, got %s' % (
      string.join(map(str, self.needed[:-1]), ', '),
      str(self.needed[-1]), str(self.given)
    )

class PXTLRuntimeError(PXTLError):
  """ An error occurred whilst executing Python code from a template. 'trace'
      is the traceback of the root-cause error.
  """
  def __init__(self, context, exc_info):
    PXTLError.__init__(self, context)
    self.error= exc_info[1]
    self.trace= exc_info[2]
  def describeError(self):
    return '%s: %s' % (str(self.error.__class__), self.error)

# Specific expections
#
class PXTLParseError(PXTLError):
  """ A template couldn't be parsed by the DOM implementation. If it is pxdom,
      we can read the location of the parse error. 
  """
  def __init__(self, error):
    context= NullNode()
    try:
      context.pxdomLocation= error.location
    except AttributeError:
      pass
    PXTLError.__init__(self, context)
    self.error= error
  def describeError(self):
    return '%s: %s' % (str(self.error.__class__), str(self.error))
  def describeContext(self):
    try:
      if self.context.pxdomLocation.uri is not None:
        return 'template '+self.context.location.uri
    except AttributeError:
      pass
    return 'template'

class PXTLImportSrcError(PXTLError):
  def __init__(self, context, src):
    PXTLError.__init__(self, context)
    self.src= src
  def describeError(self):
    return 'could not open URI %s' % self.src

class PXTLDocumentTypeError(PXTLTypeError):
  def __init__(self, context, given):
    PXTLTypeError.__init__(self, context, given, types.TupleType)

class PXTLMethodValueError(PXTLValueError):
  def __init__(self, given):
    PXTLValueError.__init__(self, None, given, ('xml','xhtml','html','text'))

class PXTLTagNameError(PXTLNameError):
  def __init__(self, context):
    PXTLNameError.__init__(self, context, context.nodeName)

class PXTLAttributeNameError(PXTLNameError):
  def __init__(self, context):
    PXTLNameError.__init__(self, context, context.nodeName)

class PXTLAttributeRootError(PXTLAttributeNameError):
  def describeError(self):
    return 'Attribute %s may only be used on root' % self.context.nodeName

class PXTLAttributeActiveError(PXTLAttributeNameError):
  def describeError(self):
    return 'Attribute %s may only be used on literals' % self.context.nodeName
  
class PXTLPITargetNameError(PXTLNameError):
  def __init__(self, context):
    PXTLNameError.__init__(self, context, context.nodeName)

class PXTLNoParserError(PXTLSyntaxError):
  def describeError(self):
    return 'a DOM3 Load implementation is needed to parse imported templates'

class PXTLAttributeMissingError(PXTLSyntaxError):
  def __init__(self, context, name):
    PXTLSyntaxError.__init__(self, context)
    self.name= name
  def describeError(self):
    return '%s required' % self.name

class PXTLMultipleConditionalError(PXTLSyntaxError):
  def describeError(self):
    return 'each element may only have one PXTL conditional attribute'

class PXTLTagnameAttrError(PXTLTypeError):
  def __init__(self, attr, given):
    PXTLTypeError.__init__(self, attr, given, types.StringType)

class PXTLTagnamespaceAttrError(PXTLValueError):
  def __init__(self, attr, given):
    PXTLValueError.__init__(self, attr, given, ['anything but PXTL or xmlns'])

class PXTLForItemsError(PXTLSyntaxError):
  def describeError(self):
    return "for element must have exactly one 'in' or 'range' attribute"

class PXTLForRangeError(PXTLTypeError):
  def __init__(self, attr, given):
    PXTLTypeError.__init__(self, attr, given, types.TupleType)

class PXTLWhileMinError(PXTLTypeError):
  def __init__(self, attr, given):
    PXTLTypeError.__init__(self, attr, given, types.IntType)

class PXTLPseudoPIOpenError(PXTLSyntaxError):
  def __init__(self, attr, ix):
    PXTLSyntaxError.__init__(self, attr)
    self.ix= ix
  def describeError(self):
    return 'pseudo-PI left open, beginning %s...' % (
      self.context.nodeValue[self.ix:self.ix+9]
    )

class PXTLPseudoPITargetNameError(PXTLNameError):
  def __init__(self, context, name):
    PXTLNameError.__init__(self, context, name)
  def describeError(self):
    return 'unknown PXTL pseudo-PI target %s' % self.name

class PXTLPseudoPITargetPXError(PXTLNameError):
  def describeError(self):
    return 'Expected PXTL target name in Pseudo-PI, found \'%s\'' % self.name

class PXTLAttrTypeError(PXTLTypeError):
  def __init__(self, attr, given):
    PXTLTypeError.__init__(self, attr, given, types.DictType)

class PXTLAttrNameTypeError(PXTLTypeError):
  def __init__(self, attr, given):
    PXTLTypeError.__init__(self, attr, given, types.StringType)

class PXTLFutureFeatureError(PXTLSyntaxError):
  def __init__(self, context, feature):
    PXTLSyntaxError.__init__(self, context)
    self.feature= feature
  def describeError(self):
    return 'feature %s unimplemented in this version of Python' % self.feature

class PXTLCallTypeError(PXTLTypeError):
  def __init__(self, context, given):
    PXTLTypeError.__init__(self, context, given, types.InstanceType)
  def describeError(self):
    return 'call fn must be an object defined by the def element, not %s'

class PXTLCallArgsError(PXTLValueError):
  def __init__(self, context, given):
    PXTLValueError.__init__(self, context, given, types.NoneType)
  def describeError(self):
    return 'subtemplate function does not take arguments'

class PXTLConditionalOrderError(PXTLSyntaxError):
  def __init__(self, context, old_cond):
    PXTLSyntaxError.__init__(self, context)
    self.old_cond= old_cond
  def describeError(self):
    if self.old_cond is None:
      return '%s must follow a conditional clause' % self.context.nodeName
    return '%s cannot follow %s' % (self.context.nodeName, self.old_cond)

class PXTLImportGlobalsError(PXTLTypeError):
  def __init__(self, context, given):
    PXTLTypeError.__init__(self, context, given, types.DictType)

class OutputMismatch(Exception):
  """ A compiled template was asked to output with a different method to what
      it had at compile-time. Not an error condition, this is caught by the
      optimised implementation, causing a recompile.
  """
  pass
