""" Provide patched-up versions of common Python DOM implementations. PXTL
    requires parts of DOM Level 3 Core and Load which are not yet implemented
    in current versions of these implementations, and it also gets tripped up
    by certain bugs in their DOM Level 2 Core code. This module patches them
    to a level PXTL can mostly work with.

    DEPRECATED as it is too unreliable. pxtl.fixdom will be removed in a
    future release of pxtl.
"""

__all__= [
  'getImplementation'
]

import urllib, urlparse
from pxtl.constants import *

_implementations= {}

def getImplementation(name):
  """ Return the DOMImplementation object for a fixed version of known
      implementations 'minidom' and '4DOM'. Cache newly-created
      implementations to try to avoid creating multiple implementation
      objects.
  """
  if _implementations.has_key(name):
    return _implementations[name]
  if name.lower()=='4dom':
    implementation= FourDOMImplementation()
  elif name.lower()=='minidom':
    implementation= MinidomImplementation()
  else:
    raise KeyError('Implementation %s not known. Try 4DOM or minidom.' % name)
  _implementations[name]= implementation
  return implementation


class FakeDOMConfiguration:
  """ A object with DOMConfiguration's interface, that does nothing.
  """
  def setParameter(self, name, value):
    pass
  def getParameter(self, name):
    return None
  def canSetParameter(self, name, value):
    return True
fakeDOMConfiguration= FakeDOMConfiguration()


class L2Implementation:
  """ An object delegating to a Level 2 DOMImplementation, implementing a DOM
      Level 3 LSParser on top of it. Subclasses must pass a DOMImplementation
      object to the constructor, and provide a function 'getDocument' that
      returns a DOM tree for a given URI.
  """
  [MODE_SYNCHRONOUS, MODE_ASYNCHRONOUS]= range(1, 3)
  domConfig= fakeDOMConfiguration

  def __init__(self, implementation):
    self.implementation= implementation
    if not implementation.__dict__.has_key('__fixdom_patched'):
      implementation.__dict__['__fixdom_patched']= True
      self.patchModule()
  def hasFeature(self, a1, a2):
    return self.implementation.hasFeature(a1, a2)
  def getFeature(self, a1, a2):
    return self.implementation.getFeature(a1, a2)
  def createDocument(self, a1, a2, a3):
    return self.implementation.createDocument(a1, a2, a3)
  def createDocumentType(self, a1, a2, a3):
    return self.implementation.createDocumentType(a1, a2, a3)

  def createLSParser(self, mode, schemaType):
    return self
  def parseURI(self, uri):
    doc= self.getDocument(uri, self)
    doc.documentURI= uri
    self.fixNode(doc.documentElement)
    return doc

  def patchModule(self):
    """ Can be extended by subclasses to perform crude run-time bug-fixing
        surgery on the implementation classes they are going to be using, the
        first time that the implementation module is loaded.
    """
    pass

  def fixNode(self, node):
    """ Can be extended by subclasses to walk the parsed DOM tree before it
        is returned, fixing up known errors in the output.
    """
    pass


class FourDOMImplementation(L2Implementation):
  """ Fixed version of PyXML's 4DOM.
  """
  NOIMP= ('4DOM is not available. If you wish to use FourDOMImplementation, '
    'please install PyXML 0.7 or later.'
  )
  def __init__(self):
    try:
      from xml.dom.DOMImplementation import implementation
      from xml.dom.ext.reader import PyExpat
      import _xmlplus
    except ImportError:
      raise ImportError(self.NOIMP)
    if _xmlplus.version_info<(0,7):
      raise ImportError(self.NOIMP)
    self.reader= PyExpat.Reader
    L2Implementation.__init__(self, implementation)

  def patchModule(self):
    L2Implementation.patchModule(self)
    from xml.dom.Document import Document
    from xml.dom.Element import Element

    # Add documentURI attribute
    #
    if not Document._readComputedAttrs.has_key('documentURI'):
      Document._get_documentURI= document_get_documentURI
      Document._set_documentURI= document_set_documentURI
      Document._readComputedAttrs['documentURI']= document_get_documentURI
      Document._writeComputedAttrs['documentURI']= document_set_documentURI

    # Add XML declaration attributes
    #
    if not Document._readComputedAttrs.has_key('xmlVersion'):
      Document._get_xmlVersion= document_get_xmlVersion
      Document._set_xmlVersion= document_set_xmlVersion
      Document._get_xmlStandalone= document_get_xmlStandalone
      Document._get_xmlEncoding= document_get_xmlEncoding
      Document._get_inputEncoding= document_get_inputEncoding
      Document._readComputedAttrs['xmlVersion']= document_get_xmlVersion
      Document._writeComputedAttrs['xmlVersion']= document_set_xmlVersion
      Document._readComputedAttrs['xmlStandalone']= document_get_xmlStandalone
      Document._readComputedAttrs['xmlEncoding']= document_get_xmlEncoding
      Document._readComputedAttrs['inputEncoding']= document_get_inputEncoding

    # Add lookupNamespaceURI method.
    #
    if not hasattr(Element, 'lookupNamespaceURI'):
      Element.lookupNamespaceURI= element_lookupNamespaceURI

  def getDocument(self, uri, imp):
    doc= self.reader().fromUri(uri)
    doc.__dict__['__implementation']= imp
    return doc

  def fixNode(self, node):
    """ Fix bugs in 4DOM trees. 4DOM gets prefix and localName the wrong way
        around in default namespace declarations.
    """
    if node.attributes is not None:
      for i in range(node.attributes.length):
        d= node.attributes.item(i).__dict__
        if d['__localName'] is None and d['__prefix']=='xmlns':
          d['__localName']= 'xmlns'
          d['__prefix']= None
    for child in node.childNodes:
      self.fixNode(child)


class MinidomImplementation(L2Implementation):
  """ Fixed version of Python/PyXML's minidom module.
  """
  NOIMP= ('minidom is not available. If you wish to use MinidomImplementation'
    ', please install Python 2.2 or later, or PyXML 0.7 or later.'
  )
  def __init__(self):
    try:
      from xml.dom import minidom
    except ImportError:
      raise ImportError(self.NOIMP)
    testdoc= minidom.parseString('<x x="x"/>')
    if testdoc.documentElement.getAttributeNode('x').namespaceURI is not None:
      raise ImportError(self.NOIMP)
    L2Implementation.__init__(self, minidom.getDOMImplementation())

  def patchModule(self):
    L2Implementation.patchModule(self)
    from xml.dom import minidom, pulldom
    testdoc= minidom.parseString('<el xmlns:n="u"><?pi p?></el>')

    # Add documentURI property
    #
    if not hasattr(testdoc, 'documentURI'):
      minidom.Document._get_documentURI= document_get_documentURI
      minidom.Document._set_documentURI= document_set_documentURI
      if not hasattr(testdoc, '__getattr__'):
        setattr(minidom.Document, 'documentURI',
          property(document_get_documentURI, document_set_documentURI)
        )

    # Add XML declaration properties
    #
    if not hasattr(testdoc, 'xmlVersion'):
      minidom.Document._get_xmlVersion= document_get_xmlVersion
      minidom.Document._set_xmlVersion= document_set_xmlVersion
      minidom.Document._get_xmlStandalone= document_get_xmlStandalone
      minidom.Document._get_xmlEncoding= document_get_xmlEncoding
      minidom.Document._get_inputEncoding= document_get_inputEncoding
      if not hasattr(testdoc, '__getattr__'):
        setattr(minidom.Document, 'xmlVersion',
          property(document_get_xmlVersion, document_set_xmlVersion)
        )
        setattr(minidom.Document, 'xmlEncoding',
          property(document_get_xmlEncoding)
        )
        setattr(minidom.Document, 'xmlStandalone',
          property(document_get_xmlStandalone)
        )
        setattr(minidom.Document, 'inputEncoding',
          property(document_get_inputEncoding)
        )

      # Add lookupNamespaceURI method
      #
      if not hasattr(testdoc.documentElement, 'lookupNamespaceURI'):
        minidom.Element.lookupNamespaceURI= element_lookupNamespaceURI

    # Test for lack of importNode method (Python <2.3, PyXML <0.8).
    #
    if not hasattr(testdoc, 'importNode'):
      minidom.Document.importNode= minidom_importNode
      minidom.Document.ownNodes= minidom_ownNodes

    # Test for cloneNode PI/Comment bug (PyXML 0.8.0 only).
    #
    try:
      testdoc.documentElement.firstChild.cloneNode(False)
    except NameError:
      for nodeType in ('PROCESSING_INSTRUCTION_NODE', 'COMMENT_NODE'):
        setattr(minidom, nodeType, getattr(minidom.Node, nodeType))

    # Test for expatreader xml: namespace bug (Python 2.2 only).
    #
    try:
      minidom.parseString('<el xml:lang="en"/>')
    except IndexError:
      pulldom.DOMEventStream.___init__= pulldom.DOMEventStream.__init__
      pulldom.DOMEventStream.__init__= eventstream___init__
    except KeyError:
      pass

    # Test for pulldom xml: namespace bug (Python <2.2.3).
    #
    try:
      minidom.parseString('<el xml:lang="en"/>')
    except KeyError:
      pulldom.PullDOM.___init__= pulldom.PullDOM.__init__
      pulldom.PullDOM.__init__= pulldom___init__

    # Test for lack of removeAttribute tolerance (all current versions).
    #
    try:
      testdoc.documentElement.removeAttribute('noattr')
    except:
      minidom.Element._removeAttribute= minidom.Element.removeAttribute
      minidom.Element.removeAttribute= minidom_removeAttribute

  def getDocument(self, uri, imp):
    """ Must read content from URI manually as minidom cannot parse URIs.
    """
    from xml.dom import minidom
    resource= urllib.urlopen(uri)
    content= resource.read()
    resource.close()
    doc= minidom.parseString(content)
    doc.implementation= imp
    return doc

  def fixNode(self, node):
    """ Fix bugs in the returned DOM for minidom, which sets the value of
        blank namespace declaration Attrs to None instead of ''.
    """
    if node.attributes is not None:
      for i in range(node.attributes.length):
        attribute= node.attributes.item(i)
        if attribute.value is None:
          attribute.value= ''
    for child in node.childNodes:
      self.fixNode(child)


# The following functions are not run directly, but are copied into the DOM
# implementation modules where they are needed.

# Implement DOM L3 Core's documentURI property. Copied into minidom.Document
# and Document.Document.
#
def document_get_documentURI(self):
  return self.__dict__.get('__fixdom_documentURI', None)
def document_set_documentURI(self, value):
  self.__dict__['__fixdom_documentURI']= value

# Implement DOM L3 Core's xml declaration properties. Copied into
# minidom.Document and Document.Document.
#
def document_get_xmlVersion(self):
  return self.__dict__.get('__fixdom_xmlVersion', '1.0')
def document_set_xmlVersion(self, value):
  self.__dict__['__fixdom_xmlVersion']= value
def document_get_xmlStandalone(self):
  return False
def document_get_xmlEncoding(self):
  return None
def document_get_inputEncoding(self):
  return 'utf-8'

def element_lookupNamespaceURI(self, prefix):
  """ Implement namespace lookup.  Copied into minidom.Element and
      Element.Element. This should be on other nodes too, but pxdom only uses
      it on elements.
  """
  if self.namespaceURI is not None and self.prefix==prefix:
    return self.namespaceURI
  xmlns= self.getAttributeNodeNS(NSNS, prefix or 'xmlns')
  if xmlns is not None:
    return xmlns.value or None
  if self.parentNode is None or self.parentNode.nodeType!=self.ELEMENT_NODE:
    return None
  return self.parentNode.lookupNamespaceURI(prefix)

def minidom_importNode(self, importedNode, deep):
  """ Implement the DOM 2 Core importNode method. Copied into
      xml.dom.minidom.Document.
  """
  t= importedNode.nodeType
  if t in [self.DOCUMENT_NODE, self.DOCUMENT_TYPE_NODE]:
    from xml.dom import NotSupportedErr
    raise NotSupportedErr()

  node= importedNode.cloneNode(deep or t==self.ATTRIBUTE_NODE)
  self.ownNodes(node)
  node.parentNode= None
  if t==self.ATTRIBUTE_NODE:
    node.ownerElement= None
  return node

def minidom_ownNodes(self, node):
  """ Set the ownerDocument of a node and all its child and attribute nodes.
      Recursive helper method used by importNode. Copied into
      xml.dom.minidom.Document.
  """
  node.ownerDocument= self
  if node.attributes!=None:
    for i in range(node.attributes.length):
      self.ownNodes(node.attributes.item(i))
  for child in node.childNodes:
    self.ownNodes(child)

def minidom_removeAttribute(self, name):
  """ Remove an attribute without complaining if we can't.
  """
  try:
    self._removeAttribute(name)
  except:
    pass

def pulldom___init__(self, documentFactory= None):
  """ Construct a pulldom that knows about the built-in namespaces. Copied
      into xml.dom.pulldom.PullDOM. Uses the old constructor stored as
      ___init__.
  """
  self.___init__(documentFactory)
  self._ns_contexts[-1][XMNS]= 'xml'

def eventstream___init__(self, stream, parser, bufsize):
  """ Construct a DOMEventStream after checking if the parser supplied is a
      broken xml.sax.expatreader that blows up when presented with an xml:
      namespace. If it is, hack its namespace stack so it doesn't. Copied into
      xml.dom.pulldom.DOMEventStream. Uses the old constructor stored as
      ___init__.
  """
  if hasattr(parser, '_ns_stack') and parser._ns_stack==[]:
    parser._ns_stack.append({XMNS: ['xml']})
  self.___init__(stream, parser, bufsize)
