""" Optimised implementation of the Python XML Templating Language.
"""

__all__= ['processFile', 'compileFile', 'prepareGlobals', 'Compiler', 'PXTLObject']

import os, sys, stat, string, urllib,urlparse, mimetypes, struct, marshal, imp
import pxtl, pxtl.optimised, pxtl.exceptions, pxtl.constants, pxtl.serialisers
from pxtl.constants import *
from pxtl.exceptions import *
from pxtl.utilities import *

BYTECODE_DIR_PERMS= 0750


def processFile(
  path, writer= None, globals= None, debug= False, headers= False, dom= None,
  bytecodebase= {}, _method= None, _encoding= None
):
  """ Compile a template into Python bytecode (pxc) or load an existing cached
      bytecode file. Execute it with optional scope.

      writer  - stream-like object to send serialised output (default stdout)
      globals - dictionary to use as template global scope (default {})
      debug   - output a debugging page if there's an error (default False)
      headers - output HTTP response headers (for CGI scripts, default False)
      dom     - DOM L3 Core/XML/Load implementation to use (default pxdom)
      bytecodebase - mapping of source folders to bytecode store folders
  """
  # _method - the output method being used to serialise the caller template;
  # _encoding - output encoding being used in the caller template,
  # if this call to processFile is the result of an import element. This
  # overrides any method/encoding declared in the imported template. If it is
  # None, this is the outermost processFile call.
  #
  if dom is None:
    from pxtl import pxdom
    dom= pxdom.getDOMImplementation('')
  if globals is None:
    globals= {}

  # If debug is on, page might be replaced by the error page. In this case we
  # must not produce direct output, as this could result in a non-well-formed
  # or unreadably-encoded result. When debugging, cache the output in a buffer
  # to avoid this danger.
  #
  if writer is None:
    writer= sys.stdout
  buffer= None
  if debug:
    buffer= StringIO()

  # Set up parameters on the global scope that are handled by the compiled
  # template code.
  #
  baseuri= 'file:'+urllib.pathname2url(os.path.abspath(path))
  isImported= prepareGlobals(globals,
    buffer or writer, baseuri, headers, dom, bytecodebase, _method, _encoding
  )
  try:

    # See if there is an up-to-date bytecode file; if so, grab the code from it.
    #
    code= pathc= None
    if bytecodebase is not None:
      stamp= os.stat(path)[stat.ST_MTIME]
      pathc= mapPath(path, bytecodebase)+'c'
      if os.path.exists(pathc):
        f= open(pathc, 'rb')
        if f.read(4)==imp.get_magic():
          if struct.unpack('<I', f.read(4))[0]>=stamp:
            code= marshal.load(f)
        f.close()

    # Try to run the code if we have it. If the code doesn't run due to having
    # been compiled with a different output method, or if we don't have code
    # to run, compile the template anew and execute the new code.
    #
    done= False
    if code is not None:
      done= executeCode(code, globals)
    if not done:
      code= compileFile(path, pathc, globals, dom, _method, _encoding)
      executeCode(code, globals)

  # If an error occurred in the template, and we're in debug mode, produce
  # an error page. (Debug mode is never set on calls caused by import elements
  # so errors should fall through to the outermost call.)
  #
  except PXTLError, e:
    if debug:
      processFile(ERROR_PAGE, writer,
        {'e': e, 'partial': buffer.getvalue()},
      False,headers,dom)
    else:
      raise
  else:
    if debug:
      writer.write(buffer.getvalue())

  # Finished. If finished, remove the PXTLObject from globals so the scope can
  # be reused without confusing us into ignoring the px:doctype.
  #
  if not isImported:
    del globals['__pxtl_object']


def compileFile(path,pathc,globals=None,dom=None,method=None,encoding=None):
  """ Compile a PXTL template using a Compiler then get Python-compiled
      bytecode for it and store to pathc as a standand Python bytecode file.
  """
  # Same defaults as for processFile.
  #
  if dom is None:
    from pxtl import pxdom
    dom= pxdom.getDOMImplementation('')
  if globals is None:
    globals= {}

  # Get document as DOM tree.
  #
  try:
    parser= dom.createLSParser(dom.MODE_SYNCHRONOUS, None)
  except NameError:
    raise PXTLNoParserError(None)
  parser.domConfig.setParameter('entities', True)
  parser.domConfig.setParameter('cdata-sections', True)
  if parser.domConfig.canSetParameter('pxdom-resolve-resources', False):
    parser.domConfig.setParameter('pxdom-resolve-resources', False)
  uri= 'file:'+urllib.pathname2url(os.path.abspath(path))
  try:
    document= parser.parseURI(uri)
  except NonErrors:
    raise
  except Exception, e:
    raise PXTLParseError(e)

  # Compile DOM tree.
  #
  compiler= Compiler()
  program= compiler.compileDocument(document, globals, method, encoding)
  code= compile(program, path, 'exec')
  if compiler.linetrans!=[1]:
    code= renumberCode(code, compiler.linetrans)

  # Try to save the compiled version to the bytecode path.
  #
  if pathc is not None:
    if not os.path.exists(os.path.dirname(pathc)):
      os.makedirs(os.path.dirname(pathc), BYTECODE_DIR_PERMS)
    stamp= os.stat(path)[stat.ST_MTIME]
    try:
      f= open(pathc, 'wb')
      try:
        f.write(imp.get_magic())
        f.write(struct.pack('<I', stamp))
        marshal.dump(code, f)
      finally:
        f.close()
    except IOError, e:
      pass
  return code


def prepareGlobals(globals,
  writer= None, baseuri= '', headers= False, dom= None, bytecodebase= None,
  _method= None, _encoding= None
):
  """ Set up a global scope suitable for executing template code in. Return
      if the scope was already in use, ie. we have been called from the import
      code of a template.
  """
  isImported= globals.has_key('__pxtl_object')
  if not isImported:
    globals['__pxtl_object']= PXTLObject(beta= False)
  globals['pxtl']= globals['__pxtl_object']
  globals['__pxtl']= pxtl
  globals['__pxtl_writer']= writer or sys.stdout
  globals['__pxtl_headers']= headers
  globals['__pxtl_dom']= dom
  globals['__pxtl_baseuri']= baseuri
  globals['__pxtl_bytecodebase']= bytecodebase
  globals['__pxtl_method']= _method
  globals['__pxtl_encoding']= _encoding
  return isImported


def executeCode(code, globals):
  """ Try to run a template with a pre-set-up global scope. Return True if it
      succeeds, False if the template has changed output method and needs
      recompiling, throw an exception if there's a general error.
  """
  try:
    exec code in globals
  except OutputMismatch:
    return False
  except NonErrors+(PXTLError,):
    raise
  except:
    raise PXTLRuntimeError(globals['__pxtl_object'], sys.exc_info())
  else:
    return True


class Compiler:
  def compileDocument(self,document,globals=None, method=None, encoding=None):
    """ Walk a PXTL document node, converting its contents into literal-output
        and logic-containing commands. Return a string containing a complete
        Python program. The 'globals' argument is only needed for calculating
        the output method (second member of the doctype tuple) - this is fixed
        at the time of compilation for efficiency, and is readable as the
        'method' property of the Compiler after compileDocument is called. The
        method calculation can be overridden by passing a method string in.
    """
    # The _writer buffer holds the final Python program output. The _queue
    # buffer holds literal XML content that the program will write when run,
    # and can be built up over content (elements, text etc.) that has no
    # active logic. When some other Python code is needed, the _queue content
    # is flushed into the program being generated before adding the code.
    #
    self._writer= StringIO()
    self._queue= TrunkIO()
    self._indent= 0
    self._emptydent= False

    # Space-attribute-handling. _space is True if the current space model is
    # to preserve, False if it's to remove, and None if the innermost-non-None
    # px:space attribute is not a static (literal) value, in which case we
    # have to evaluate it at run-time. _fnspace is a count of number of
    # functions with retained whitespace mode values, to generate unique names
    # for the storage variables that retain the inherited whitespace modes.
    #
    self._space= True
    self._fnspace= 0

    # linetrans is a mapping list from destination-program line numbers to
    # the corresponding line in the source template. This is available to the
    # caller after the method is finished, and can be used to fix up line
    # number reporting for errors.
    #
    self.linetrans= [1]

    # The nsstack is one of the trickiest bits to grok! It contains a list of
    # frames; a new frame is added each time an element is skipped, or an
    # element with conditional attribute is possibly-skipped, or a function
    # definition is entered. The frames are popped back off the stack when
    # these contexts are exited again.
    #
    # Each frame is a list of (attr, levels, callns) tuples, representing
    # namespace declarations that we may need to add to literal elements
    # in this context, if they don't themselves override that ns prefix. attr
    # is a DOM Attr node with the relevant name and value. Levels is a list of
    # previously created conditional-attribute nest-levels to check; if any of
    # these evaluated to 'true' at run-time we know that the element
    # containing that conditional attribute was not skipped, so it will
    # already have output the namespace declaration and we don't need to
    # output it again. callns is a flag; if True then the declaration might be
    # inherited from the parent of a px:call that called the subtemplate
    # definition we are currently in, so check the run-time __pxtl_callns list
    # to see if it has already been declared.
    #
    self._nsstack= [ [] ]

    # Nest-level indicators, used to create local variable names, incremented
    # as blocks using them are nested to avoid clashing.
    #
    self._n_tagname=self._n_elcond=self._n_atcond=self._n_index=self._n_min= 0
    self._haveElcond= False

    # Semi-prepare the globals, just enough for the doctype tuple to be
    # calculated, in case this is a compile-only run where we don't have a
    # full globals environment.
    #
    if globals is None:
      globals= {}
    if not globals.has_key('__pxtl_object'):
      globals['pxtl']= PXTLObject(beta= False)
    self.globals= globals

    # Ensure any future-features are available
    #
    root= document.documentElement
    future= root.getAttributeNodeNS(PXNS, 'future')
    features= checkFeatures(future)
    if len(features)>0:
      self._write('from __future__ import %s' % ', '.join(features))

    # Evaluate the doctype tuple. Whilst doing this for real, write code into
    # the program that will do it at run-time too. Note, assignFrom's context
    # is None because the doctype attribute may occur after other content in
    # document order even though it is calculated first. If we passed in the
    # context we could thus confuse the linetrans lookup table.
    #
    doctype= root.getAttributeNodeNS(PXNS, 'doctype')
    doctuple= (None,)*4
    if doctype is None:
      self._write('__pxtl_v= (__pxtl.constants.none,)*5')
    else:
      self.assignFrom(None, doctype.nodeValue)
      self._write('__pxtl_v= (__pxtl_v+(__pxtl.constants.none,)*5)[:5]')
      if method is None:
        doctuple= self.pxeval(doctype.value, self.globals, doctype)
        if type(doctuple)!=type(()):
          raise PXTLDocumentTypeError(doctype, doctuple)

    # Work out the method from the doctype tuple, root node and the 'method'
    # argument override. Again, have the run-time program do the same.
    #
    if method is None:
      method= determineMethod(doctuple, root.namespaceURI, root.localName)
      if not pxtl.serialisers.SERIALISERS.has_key(method):
        raise PXTLMethodValueError(method)
    self.method= method
    self._write('if __pxtl_method is __pxtl.constants.none:', dent= 1)
    self._write('__pxtl_method= __pxtl.utilities.determineMethod('
      '__pxtl_v, %s, %s)' % (repr(root.namespaceURI), repr(root.localName)),
    dent= -1)

    # Work out the encoding from the template's encoding and the 'encoding'
    # argument override.
    #
    if UnicodeType is None:
      self.encoding= None
    elif encoding is not None:
      self.encoding= encoding
    else:
      self.encoding= document.inputEncoding or document.xmlEncoding or 'utf-8'

    # Have the run-time program check that the method/encoding it has guessed
    # matches what we have calculated during compile-time. If not, the output
    # would likely be malformed, so throw an exception causing a re-compile.
    # Also check the pxtl version number has not changed - may need to
    # recompile bytecode if pxtl itself is upgraded.
    #
    self._write((
      'if __pxtl_method!=%s or __pxtl_encoding not in '
      '(__pxtl.constants.none, %s) or __pxtl.__version__!=%s:'
    ) % (repr(method), repr(encoding), repr(pxtl.__version__)), dent= 1)
    self._write('raise __pxtl.exceptions.OutputMismatch()', dent= -1)

    # Make a serialiser for the correct method and encoding. Have the run-time
    # do the same.
    #
    self._serialiser= pxtl.serialisers.SERIALISERS[method](self._queue, self.encoding)
    self._write('__pxtl_serialiser= __pxtl.serialisers.SERIALISERS'+
      '[%s](__pxtl_writer, %s)' % (repr(method), repr(self.encoding))
    )
    self._write('__pxtl_object.serialiser= __pxtl_serialiser')

    # Get the program to write a HTTP Content-Type header if required.
    #
    self._write('if __pxtl_headers:', dent= 1)
    self._write( '__pxtl.utilities.writeTypeHeader(__pxtl_writer, '
      '__pxtl_v[0], __pxtl_method, %s)' % repr(self.encoding), dent= -1
    )

    # Store the doctype details in global context, unless there is already
    # something there, in which case we are being imported and we should leave
    # it alone. Set a flag stating that so far we have maybe-not written the
    # doctype (if there is one - even if there is no doctype attribute on this
    # template there might be one from a template that imported us). This will
    # be checked when the first element (output root element) is written.
    #
    self._write('if __pxtl_object.doctype is __pxtl.constants.none:', dent=1)
    self._write('__pxtl_object.doctype= __pxtl_v[2:5]')
    self._checkDoctype= True
    self._serialiser.startDocument(document)
    self._write(dent= -1)

    # Recursively compile children. When finished, clear any remaining literal
    # output and return the complete buffer.
    #
    self._write('__pxtl_spaces= []')
    self.compileChildren(document)
    self._write()
    return self._writer.getvalue()


  # Compilation output helper methods
  #
  def _write(self, line= None, context= None, dent= 0, lineno= 0):
    """ Write a line of code to the output program buffer, dealing with
        indenting, queueing, context-setting and line renumbering issues.
    """
    # Flush any literal output content waiting in the queue.
    #
    queue= self._queue.getvalue()
    if queue!='':
      self._queue.truncate(0)
      self._write('__pxtl_writer.write(%s)' % repr(queue))
    indent= '  '*self._indent

    # If there is a context node, write a command to note its position in the
    # original template.  This error-pinpointing option isonly available when
    # the DOM implementation we are using is pxdom - other DOMs do not allow
    # one to read the row/column position of arbitrary nodes.
    #
    lineNumber= None
    if context is not None and hasattr(context, 'pxdomLocation'):
      lineNumber= context.pxdomLocation.lineNumber
      if lineno<2:
        if context.pxdomLocation.uri is None:
          uri= '__pxtl.constants.none'
        else:
          uri= repr(context.pxdomLocation.uri)
        self._writer.write('%s__pxtl_object.context(%d,%d,%s,%s,%s)\n' % (indent,
          context.pxdomLocation.columnNumber, lineNumber, uri,
          repr(context.nodeType), repr(context.nodeName)
        ))
        self._emptydent= False
      lineNumber= lineNumber+lineno

    # Maintain the lookup list of program-to-template line numbers, for
    # accurate traceback reporting (again, pxdom only).
    #
    if lineNumber is not None:
      progline= string.count(self._writer.getvalue(),'\n')+1
      while len(self.linetrans) < progline:
        self.linetrans.append(self.linetrans[-1])
      self.linetrans.append(lineNumber)

    # Output the command itself.
    #
    if line is not None:
      self._writer.write('%s%s\n' % (indent, line) )
      self._emptydent= False

    # Change indentation level. If we are unindenting but have not printed a
    # line of code since starting the indent, we have to put a pass statement
    # in to stop it being a syntax error.
    #
    if dent<0 and self._emptydent:
      self._writer.write('%spass\n' % indent)
      self._emptydent= False
    if dent>0:
      self._emptydent= True
    self._indent= self._indent+dent
    if self._indent<0:
      raise IndentationError('should not happen')


  def assignFrom(self, node, expr= None, var= 'v', truth= False):
    """ Write to the program an assignment line writing an expression to a
        PXTL internal run-time variable. If an expr argument is not passed,
        the expression is obtained from the value of the context node. The
        variable used is __pxtl_v unless a var argument is passed, in which
        case its name is used, prefixed by __pxtl_. If truth is set, the bool
        value will be assigned.

        Check that we can compile the expression on its own now so we can
        catch syntax errors early and not have to worry about it at the final
        compile step.
    """
    if expr is None:
      expr= node.nodeValue
    try:
      compile(expr, '<compile test>', 'eval')
    except SyntaxError:
      raise PXTLRuntimeError(node, sys.exc_info())
    if truth:
      self._write('__pxtl_%s= not not (%s)' % (var, expr), node)
    else:
      self._write('__pxtl_%s= (%s)' % (var, expr), node)

  def assignTo(self, node, lvalue= None, var= 'v'):
    """ As assignFrom, but write a line assigning an internal variable's value
        to a given expression - which must be a valid l-value - instead of the
        other way around.
    """
    if lvalue is None:
      lvalue= node.nodeValue
    try:
      compile('(%s)= SomeValue' % lvalue, '<compile test>', 'exec')
    except SyntaxError:
      raise PXTLRuntimeError(node, sys.exc_info())
    self._write('(%s)= __pxtl_%s' % (lvalue, var), node)


  def pxeval(self, expr, globals, context):
    """ Wrapper for 'eval' for the rare occasions where the optimised
        implementation needs to evaluate a Python expression from a template
        at compile-time instead of run-time. Similar to - but simpler than -
        the reference implementation's Scoper.pxeval.
    """
    code= '__pxtl_v= (%s)\n' % string.lstrip(expr)
    try:
      if convertExecs:
        code= str(code)
      exec code in globals
    except NonErrors+(PXTLError,):
      raise
    except:
      raise PXTLRuntimeError(context, sys.exc_info())
    return globals['__pxtl_v']


  # Dispatchers called from compileDocument
  #
  def compileChildren(self, node):
    """ Dispatch all child nodes to the appropriate handler. Some simple node
        types that contain no dynamic content can be serialised right away.
    """
    oldCheck= self._checkDoctype

    for child in node.childNodes:
      self._serialiser.beforeChild(child, ignoreDoctypeHack= True)
      if child.nodeType==node.ELEMENT_NODE:
        self.compileElement(child)
      elif child.nodeType==node.PROCESSING_INSTRUCTION_NODE:
        self.compilePI(child)
      elif child.nodeType==node.CDATA_SECTION_NODE:
        self._serialiser.writeCDATA(child)
      elif child.nodeType==node.ENTITY_REFERENCE_NODE:
        self._serialiser.writeEntity(child)
      elif child.nodeType==node.COMMENT_NODE:
        self._serialiser.writeComment(child)

      # For text nodes, remove whitespace if _space mode is False. If it's
      # None, make the program look at the spaces stack at run-time.
      #
      elif child.nodeType==node.TEXT_NODE:
        data= string.lstrip(child.data)
        pre= child.data[:len(child.data)-len(data)]
        data= string.rstrip(data)
        post= child.data[len(pre)+len(data):]

        if self._space is not None:
          if self._space:
           self._serialiser.writeString(pre)
          self._serialiser.writeString(data)
          if self._space:
           self._serialiser.writeString(post)

        else:
          self._write('__pxtl_v= __pxtl.utilities.lastItem(__pxtl_spaces)')
          if pre!='':
            self._write('if __pxtl_v:', dent= 1)
            self._serialiser.writeString(pre)
            self._write(dent= -1)
          self._serialiser.writeString(data)
          if post!='':
            self._write('if __pxtl_v:', dent= 1)
            self._serialiser.writeString(post)
            self._write(dent= -1)

    # Restore flags where child content doesn't affect parents.
    #
    self._haveElcond= False
    self._checkDoctype= oldCheck


  def compileElement(self, node):
    """ Process an element - possibly a PXTL one, or just a literal - by
        dispatching to the appropriate method.
    """
    # Check for PXTL space attribute. If non-None, change whitespace removal
    # mode. If it's not a static expression, have the program calculate it and
    # push it onto the run-time space-values stack.
    #
    space= node.getAttributeNodeNS(PXNS, 'space')
    if space is not None:
      spaceStatic, spaceValue= evalStatic(space.value)
      if spaceStatic and spaceValue is None:
        space= None
    if space is not None:
      oldSpace= self._space
      if spaceStatic:
        self._space= not not spaceValue
      else:
        self._space= None
        if oldSpace is not None:
          self._write('__pxtl_spaces.append(%s)' % repr(oldSpace), space)
        self._write('__pxtl_spaces.append(%s)' % space.value, space)

    # Check elements for unknown PXTL attributes.
    #
    for attr in attrList(node.attributes):
      if attr.namespaceURI==PXNS:
        if attr.localName not in ATTRIBUTES:
          raise PXTLAttributeNameError(attr)
        if attr.localName in ('doctype', 'future') and (
          node!=node.ownerDocument.documentElement
        ):
          raise PXTLAttributeRootError(attr)

    # Dispatch non-active content elements.
    #
    if node.namespaceURI!=PXNS or node.localName=='none':
      self._haveElcond= False
      if node.namespaceURI==PXNS and not node.hasAttributeNS(PXNS, 'tagname'):
        self.skipElement(node)
      else:
        self.literalElement(node)
    else:

      # Check active elements for disallowed PXTL attributes.
      #
      for attr in attrList(node.attributes):
        if attr.namespaceURI==PXNS and attr.localName!='space':
          if attr.localName in CONDITIONALS+['tagname', 'attr']:
            raise AttributeActiveError(attr)

      # Dispatch active content elements.
      #
      if node.localName in CONDITIONALS:
        self.conditionalElement(node)
      elif node.localName=='for':
        self.forElement(node)
      elif node.localName=='while':
        self.whileElement(node)
      else:
        self._haveElcond= False
        if node.localName=='def':
          self.defElement(node)
        elif node.localName=='call':
          self.callElement(node)
        elif node.localName=='import':
          self.importElement(node)
        else:
          raise PXTLTagNameError(node)

    # Restore old space mode if changed.
    #
    if space is not None:
      if not spaceStatic:
        self._write('__pxtl_spaces.pop()')
        if oldSpace is not None:
          self._write('__pxtl_spaces.pop()')
      self._space= oldSpace


  def skipElement(self, node, level= None):
    """ Recurse into an element's children, transferring any namespace
        declarations to each child element (if required to keep ns). If level
        n is given, then the element might or might not be skipped, depending
        on the __pxtl_atcond_n success status; in this case, make all
        namespaces we have built up depend on the lack of success at this
        level.
    """
    newlevels= ()
    if level is not None:
      newlevels= (level,)

    # Find non-redundant namespace declarations on this element.
    #
    ns= []
    names= []
    for attr in attrList(node.attributes):
      if attr.namespaceURI==NSNS and attr.value!=PXNS:
        if node.parentNode is not None:
          if node.parentNode.nodeType!=node.DOCUMENT_NODE:
            if attr.prefix is None:
              prefix= None
            else:
              prefix= attr.localName
            value= attr.value
            if value=='':
              value= None
            if node.parentNode.lookupNamespaceURI(prefix)==value:
              continue
        ns.append(attr)
        names.append(attr.name)

    # Add them to the list of current skipped declaration attributes and push
    # the combined list to stack during child compilation.
    #
    if ns!=[] or newlevels!=():
      self._nsstack.append([])
      for attr, levels, callns in self._nsstack[-2]:
        if attr.name not in names:
          self._nsstack[-1].append((attr, levels+newlevels, callns))
      for attr in ns:
        self._nsstack[-1].append((attr, newlevels, False))

    self.compileChildren(node)

    if ns!=[] or newlevels!=():
      del self._nsstack[-1]


  def compilePI(self, node):
    """ Dispatch processing instructions to the appropriate method if they are
        PXTL PIs; send other PIs straight to the serialiser.
    """
    target= None
    if SHORTCUTS.has_key(node.target):
      target= SHORTCUTS[node.target]
    elif node.target[:3]=='px_':
      target= node.target[3:]
    elif node.target[:3]=='px:':
      raise PXTLPITargetNameError(node)
    if target is not None:
      if target=='code':
        self.blockPI(node)
      elif target=='note':
        pass
      elif target in TARGETS:
        self.expressionPI(node, target)
      else:
        raise PXTLPITargetNameError(node)
    else:
      self._serialiser.writePI(node)


  # Node handlers called by dispatchers
  #
  def blockPI(self, node):
    """ Compile a px_code PI, either a single line or a block of code with
        consistent indentation. Allow pxtl.write() to send output to the final
        output.
    """
    self._write('__pxtl_object.isBlock= 1')

    # If it's a single line, test we can compile it then pass it to write.
    #
    if node.data[:2]!=':\n':
      try:
        compile(node.data+'\n', '<block>', 'exec')
      except SyntaxError:
        raise PXTLRuntimeError(node, sys.exc_info())
      self._write(string.strip(node.data), node)

    # It it's a multi-line block, extract the base-unindented logical lines
    # to pass to write(), which will add suitable indentation for the current
    # output point. The compilation test happens as part of getLogicalLines,
    # which has to use compile() a lot to work anyway.
    #
    else:
      try:
        lines= getLogicalLines(node.data[2:])
      except SyntaxError:
        raise PXTLRuntimeError(node, sys.exc_info())
      lineno= 1
      for line in lines:
        self._write(line, context= node, lineno= lineno)
        lineno= lineno+string.count(line, '\n')+1

    self._write('__pxtl_object.isBlock= 0')


  def expressionPI(self, node, coding):
    """ Compile a single-expression-with-possible-encoding PI.
    """
    self.assignFrom(node)
    self._write(
      'if __pxtl.constants.type(__pxtl_v) not in __pxtl.constants.StringTypes:',
      dent=1
    )
    self._write('__pxtl_v= __pxtl.constants.str(__pxtl_v)', dent= -1)
    if coding not in ('text', 'mark'):
      self._write('__pxtl_v=__pxtl.utilities.encodeText(__pxtl_v, %s)'
        % repr(coding)
      )
    if coding=='mark':
      self._write('__pxtl_serialiser.w(__pxtl_v)')
    else:
      self._write('__pxtl_serialiser.writeString(__pxtl_v)')


  def pseudoPI(self, target, data, attr= None):
    """ Compile an expression pseudo-PI inside an attribute value.
    """
    if SHORTCUTS.has_key(target):
      coding= SHORTCUTS[target]
    elif target[:3]=='px_' and target!='px_mark':
      coding= target[3:]
    else:
      raise PXTLPseudoPITargetPXError(attr, target)
    if coding=='note':
      return
    elif coding not in CODINGS:
      raise PXTLPseudoPITargetPXError(attr, target)

    self.assignFrom(attr, data)
    self._write(
      'if __pxtl.constants.type(__pxtl_v) not in __pxtl.constants.StringTypes:',
      dent=1
    )
    self._write('__pxtl_v= __pxtl.constants.str(__pxtl_v)', dent= -1)
    if coding!='text':
      self._write('__pxtl_v=__pxtl.utilities.encodeText(__pxtl_v,%s)'
        % repr(coding)
      )
    self._write('__pxtl_serialiser.inattr= 1')
    self._write('__pxtl_serialiser.writeString(__pxtl_v)')
    self._write('__pxtl_serialiser.inattr= 0')


  def literalElement(self, node):
    """ Compile a literal Element.
    """
    empty= len(node.childNodes)==0

    # Find any conditional attribute the element has.
    #
    condAttr= None
    for conditional in CONDITIONALS:
      if node.hasAttributeNS(PXNS, conditional):
        if condAttr is not None:
          raise PXTLMultipleConditionalError(node)
        condAttr= node.getAttributeNodeNS(PXNS, conditional)
    if condAttr is not None:

      # Create attribute conditional context either from new if unnested, or
      # by cloning the parent scope's context (allowing nested else etc.)
      #
      if self._n_atcond==0:
        self._write('__pxtl_atcond_0= __pxtl.utilities.ConditionalContext()')
      else:
        self._write('__pxtl_atcond_%d= __pxtl_atcond_%d.copy()' % (
          self._n_atcond, self._n_atcond-1
        ))

      # Ask the conditional context if we can be lazy. If not, test the given
      # conditional. Ask the conditional context if the element is successful
      # given the possibly-calculated condition; if not, suppress the
      # start-tag.
      #
      self._write('__pxtl_v= __pxtl.constants.none')
      if condAttr.localName!='else':
        self._write('if __pxtl_atcond_%d.getTesting(%s,__pxtl_object):' %
          (self._n_atcond, repr(condAttr.localName)), condAttr, dent= 1
        )
        self.assignFrom(condAttr, truth= True)
        self._write(dent= -1)
      self._write('if __pxtl_atcond_%d.getSuccess(%s, __pxtl_v):' %
        (self._n_atcond, repr(condAttr.localName)), dent= 1
      )

    # If a tagname attribute is used this must be calculated on the fly, and
    # stored in a variable for use later (eg. in end-tag). Otherwise we can
    # output it to the literal queue.
    #
    tagname= node.getAttributeNodeNS(PXNS, 'tagname')
    if tagname is not None:
      tagvar= 'tagname_%d' % self._n_tagname
      self._n_tagname= self._n_tagname+1
      self.assignFrom(tagname, var= tagvar)

    # If it is possible that this start-tag may be the root element, have the
    # program check whether there is a doctype declaration to output before
    # opening the start-tag.
    #
    if self._checkDoctype:
      self._write('if __pxtl_object.doctype!=(__pxtl.constants.none,)*3:',
        dent= 1
      )
      if tagname is not None:
        tag= '__pxtl_'+tagvar
      else:
        tag= repr(node.tagName)
      self._write('__pxtl_serialiser.startDoctype(%s, __pxtl_object.doctype'
        '[0], __pxtl_object.doctype[1], __pxtl_object.doctype[2], 1)' % tag
      )
      self._write('__pxtl_object.doctype=(__pxtl.constants.none,)*3',dent=-1)
    if condAttr is None:
      self._checkDoctype= False

    # Output start of start-tag.
    #
    if tagname is not None:
      self._write('__pxtl_serialiser.startElement(__pxtl_%s, %s)' % (
        tagvar, ['0', '1'][empty]
      ))
    else:
      self._serialiser.startElement(node.tagName, empty)

    # Walk across predefined attributes. If an attr attribute is defined, the
    # run-time program will need to skip over attributes whose names are in
    # the attr mapping's keys.
    #
    attrAttr= node.getAttributeNodeNS(PXNS, 'attr')
    if attrAttr is not None:
      self.assignFrom(attrAttr, var= 'attr')
    for attr in attrList(node.attributes):
      self.compileAttribute(attr, attrAttr is not None)

    # If there are any namespace declarations from skipped parent elements,
    # and they aren't overridden by declarations on this element, output them
    # here. If a nonempty levels is given in the nsstack entry, that namespace
    # might already have been written, so have the program test the success of
    # the element at that atcond level before outputting.
    #
    haveNS= self._nsstack[-1]!=[]
    if haveNS:
      for attr, levels, callns in self._nsstack[-1]:
        if not node.hasAttribute(attr.name):
          if len(levels)!=0:
            self._write('if not (%s):' % string.join(
              map(lambda level: '__pxtl_atcond_%d.old_success'%level, levels),
            ' or '), dent= 1)
          if callns:
            self._write('if %s in __pxtl_callns:' % repr(attr.name), dent= 1)
          self.compileAttribute(attr, attrAttr is not None)
          if callns:
            self._write(dent= -1)
          if len(levels)!=0:
            self._write(dent= -1)

    # If this element cannot be skipped, we can be sure that we won't need to
    # move any namespace declarations to descendants. So push a new blank
    # context onto the nsstack.
    #
    newNsstack= haveNS and condAttr is None and self._nsstack[-1]!=[]
    if newNsstack:
      self._nsstack.append([])

    # If an attr attribute is defined, have the run-time program output any of
    # added/overridden attributes from the attr mapping.
    #
    if attrAttr is not None:
      self._write('for __pxtl_k, __pxtl_v in __pxtl_attr.items():', dent= 1)
      self._write('if __pxtl_v is not __pxtl.constants.none:', dent= 1)
      self._write('__pxtl_serialiser.inattr= 1')
      self._write('for __pxtl_t in __pxtl.constants.xrange('
        '__pxtl_serialiser.attributeTimes(__pxtl_k)'
      '):', dent= 1)
      self._write('__pxtl_serialiser.startAttribute(__pxtl_k, __pxtl_t)')
      self._write('__pxtl_serialiser.writeString(__pxtl_v)')
      self._write(
        '__pxtl_serialiser.endAttribute(__pxtl_k, __pxtl_t)',
      dent= -3)
      self._write('__pxtl_serialiser.inattr= 0')

    # Output the end of the start-tag. If there's a tagname override use the
    # previously calculated name. (The tagname may be needed by the XHTML
    # serialiser to decide whether to use an empty element shorthand or not.)
    #
    if tagname is not None:
      self._write('__pxtl_serialiser.endElement(__pxtl_%s, %s)' % (
        tagvar, ['0', '1'][empty]
      ))
    else:
      self._serialiser.endElement(node.tagName, empty)
    if condAttr is not None:
      self._write(dent= -1)

    # Recurse into child nodes. If this element might be skipped, use
    # skipElement to recurse, so it can update the nsstack.
    #
    if not empty:
      if condAttr is not None:
        level= self._n_atcond
        self._n_atcond= self._n_atcond+1
        self.skipElement(node, level)
        self._n_atcond= self._n_atcond-1
      else:
        self.compileChildren(node)

    # Remove a new ns context if we pushed one before the walk.
    #
    if newNsstack:
      del self._nsstack[-1]

    # Write the end-tag. If a conditional attribute was used and it was
    # unsuccessful the end-tag must also be suppressed. If a tagname was
    # used this must be got from a variable.
    #
    if condAttr is not None:
      self._write('if __pxtl_atcond_%d.old_success:' % self._n_atcond, dent=1)

    if tagname is not None:
      self._write('__pxtl_serialiser.closeElement(__pxtl_%s, %s)' % (
        tagvar, ['0', '1'][empty]
      ))
      self._n_tagname= self._n_tagname-1
    else:
      self._serialiser.closeElement(node.tagName, empty)

    if condAttr is not None:
      self._write(dent= -1)


  def compileAttribute(self, node, haveAttr= False):
      # Don't include PXTL attributes in output!
      #
      if node.namespaceURI==PXNS or (
        node.namespaceURI==NSNS and node.value==PXNS
      ):
        return

      # Split the attribute child content into entity-references, text strings
      # and pseudo-PIs. If the childNodes look like they might be misleading
      # (under minidom they are), just use the attribute value instead.
      #
      l= len(node.childNodes)
      if l==0 or (l==1 and node.childNodes[0].nodeType==node.TEXT_NODE):
        parts= parsePseudoPIs(node.value, node)
      else:
        parts= []
        for child in node.childNodes:
          if child.nodeType==node.ENTITY_REFERENCE_NODE:
            parts.append((None, child))
          else:
            parts.extend(parsePseudoPIs(child.data), child)

      # See if there is a px_if pseudo-PI. If so we'll need to insert a check
      # for the condition.
      #
      haveIf= False
      for part in parts:
        if type(part)==type(()) and part[0]=='px_if':
          haveIf= True
          self.assignFrom(node, part[1])
          self._write('if __pxtl_v:', dent=1)

      # If a PXTL attr attribute was given, check at run-time that each
      # attribute we hit is not defined in the attr mapping. If it is, skip it
      # for now, it'll get written later.
      #
      if haveAttr:
        self._write('if not __pxtl_attr.has_key(%s):'%repr(node.name), dent=1)
      for time in range(self._serialiser.attributeTimes(node.name)):
        self._serialiser.startAttribute(node.name, time)
        self._serialiser.inattr= True

        # Walk over the child content, serialising entity references and
        # literal text, and inserting code to output other pseudo-PIs.
        #
        for part in parts:
          if type(part)==type(()) and part[0] is not None:
            if part[0]!='px_if':
              self.pseudoPI(part[0], part[1], node)
          elif type(part)==type(()):
            self._serialiser.writeEntity(part[1])
          else:
            self._serialiser.writeString(part)

        # Finally, end the attribute value, and any if-blocks we have open.
        #
        self._serialiser.inattr= False
        self._serialiser.endAttribute(node.name, time)
      if haveIf or haveAttr:
        self._write(dent= 0-haveIf-haveAttr)


  # PXTL structural elements
  #
  def ensureElcond(self):
    if not self._haveElcond:
      self._write('__pxtl_elcond_%d= __pxtl.utilities.ConditionalContext()' %
        self._n_elcond
      )
      self._haveElcond= True

  def conditionalElement(self, node):
    """ Compile if/else/elif/anif/orif elements.
    """
    self.ensureElcond()

    # Evaluate condition only if we need to. Usually we ask the
    # conditionalContext whether we need to, but for efficiency we
    # short-circuit that for 'if' which is always tested and 'else' which has
    # no condition to test for.
    #
    if node.localName=='else':
      self._write('__pxtl_v= __pxtl.constants.none')
    else:
      if node.localName!='if':
        self._write('if __pxtl_elcond_%d.getTesting(%s, __pxtl_object):' %
          (self._n_elcond, repr(node.localName)), node, dent= 1
        )
      test= node.getAttributeNode('test')
      if test is None:
        raise PXTLAttributeMissingError(node, 'test')
      self.assignFrom(test, truth= True)
      if node.localName!='if':
        self._write(dent= -1)
        self._write('else:', dent= 1)
        self._write('__pxtl_v= __pxtl.constants.none', dent= -1)

    # Output children in an if block, removing the current conditional context
    # from view - if a descendant needs one it will ensureElcond() to create a
    # new one itself.
    #
    self._write('if __pxtl_elcond_%d.getSuccess(%s, __pxtl_v):' %
      (self._n_elcond, repr(node.localName)), dent= 1
    )
    self._n_elcond= self._n_elcond+1
    self._haveElcond= False
    self.skipElement(node)
    self._write(dent= -1)
    self._n_elcond= self._n_elcond-1
    self._haveElcond= True


  def forElement(self, node):
    # Read attributes.
    #
    item= node.getAttributeNode('item')
    iattr= node.getAttributeNode('in')
    rattr= node.getAttributeNode('range')
    if (rattr is None) == (iattr is None):
      raise PXTLForItemsError(node)
    index= node.getAttributeNode('index')

    # Set up vars.
    #
    self.ensureElcond()
    self._write('__pxtl_elcond_%d.getSuccess("if", 0)' % self._n_elcond)
    if iattr is not None:
      self.assignFrom(iattr)
    else:
      self._write('__pxtl_v= __pxtl.constants.xrange(%s)' % rattr.value, rattr)
    if index is not None:
      self._write('__pxtl_index_%d= 0' % self._n_index)

    # Assign item and/or index vars, set up context for child content.
    #
    self._write('for __pxtl_v in __pxtl_v:', dent= 1)
    if item is not None:
      self.assignTo(item)
    if index is not None:
      self.assignTo(index, var= 'index_%d' % self._n_index)
    self._n_elcond= self._n_elcond+1
    self._haveElcond= False
    if index is not None:
      self._n_index= self._n_index+1

    # Walk children and restore context. Loop around again, incrementing index
    # counter if needed. If needed, tell the conditional context whether the
    # loop was successful.
    #
    self.skipElement(node)
    self._n_elcond= self._n_elcond-1
    self._haveElcond= True
    if index is not None:
      self._n_index= self._n_index-1
      self._write('__pxtl_index_%d= __pxtl_index_%d+1' % ((self._n_index,)*2))
      self._write('if __pxtl_index_%d==0:' % self._n_index, dent= 1)
    self._write('__pxtl_elcond_%d.getSuccess("if", 1)' % self._n_elcond)
    if index is not None:
      self._write(dent= -1)
    self._write(dent= -1)


  def whileElement(self, node):
    # Read attributes.
    #
    test= node.getAttributeNode('test')
    if test is None:
      raise PXTLAttributeMissingError(node, 'test')
    min= node.getAttributeNode('min')
    index= node.getAttributeNode('index')

    # Begin loop structure. The __pxtl_index_n counter is only required if the
    # 'index' or 'min' attributes are in use.
    #
    self.ensureElcond()
    self._write('__pxtl_elcond_%d.getSuccess("if", 0)' % self._n_elcond)
    if index is not None or min is not None:
      self._write('__pxtl_index_%d= 0' % self._n_index)
    if min is not None:
      self.assignFrom(min, var= 'min_%d' % self._n_min)
    self._write('while 1:', dent= 1)

    # Each time around, check the loop test condition and break if it is false
    # unless 'min' is in use in which case the condition might be ignored.
    #
    if min is not None:
      self._write('if __pxtl_index_%d>=__pxtl_min_%d:' %
        (self._n_index, self._n_min), dent= 1
      )
    self.assignFrom(test)
    self._write('if not __pxtl_v:', dent= 1)
    self._write('break', dent= -1)
    if min is not None:
      self._write(dent= -1)

    # Assign loop index and establish nested context.
    #
    if index is not None:
      self.assignTo(index, var= 'index_%d' % self._n_index)
    self._n_elcond= self._n_elcond+1
    self._haveElcond= False
    if index is not None or min is not None:
      self._n_index= self._n_index+1
    if min is not None:
      self._n_min= self._n_min+1

    # Walk children and restore context.
    #
    self.skipElement(node)
    self._n_elcond= self._n_elcond-1
    self._haveElcond= True
    if index is not None or min is not None:
      self._n_index= self._n_index-1
    if min is not None:
      self._n_min= self._n_min-1

    # Loop around again. Notify the element conditional context if it needs
    # to know whether the loop as a whole was successful.
    #
    if index is not None or min is not None:
      self._write('if __pxtl_index_%d==0:' % self._n_index, dent= 1)
    self._write('__pxtl_elcond_%d.getSuccess("if", 1)' % self._n_elcond)
    if index is not None or min is not None:
      self._write(dent= -1)
      self._write('__pxtl_index_%d= __pxtl_index_%d+1' % ((self._n_index,)*2))
    self._write(dent= -1)


  # Subtemplates and external resources.
  #
  def defElement(self, node):
    args= node.getAttributeNode('args')
    fn= node.getAttributeNode('fn')
    if fn is None:
      raise PXTLAttributeMissingError(node, 'fn')

    # If the whitespace mode of the def element is non-static, have the
    # program retain the calculated whitespace mode inherited into the def
    # element.
    #
    if self._space is None:
      self._write(
        '__pxtl_fnspace_%s= __pxtl.utilities.lastItem(__pxtl_spaces)'
      % self._fnspace)

    # The function name can be any l-value including things like 'mylist[0]',
    # which would be invalid in a function def. Detect l-values that are
    # possibly not Python identifiers, and use __pxtl_v as a temporary store
    # in this case. (We don't want to do it all the time as it makes debugging
    # harder if all subtemplate functions appear as '__pxtl_v'.)
    #
    funcname= string.strip(fn.value)
    if funcname[:1] in string.letters+'_':
      for ch in funcname[1:]:
        if ch not in string.letters+string.digits+'_':
          funcname= '__pxtl_v'
          break
    else:
      funcname= '__pxtl_v'
    if args is None:
      self._write('def %s():' % funcname, node, dent= 1)
    else:
      self._write('def %s(%s):' % (funcname, args.value), args, dent= 1)

    # If the whitespace mode of the def element is non-static, make a local
    # space-values stack initialised with the retained inherited value.
    #
    if self._space is None:
      self._write('__pxtl_spaces= [__pxtl_fnspace_%s]' % self._fnspace)
      self._fnspace= self._fnspace+1

    # Read the namespace context of the caller parent and make a list of any
    # namespaces on the subtemplate that might need to be redeclared when
    # called. If there are any, bang them onto the NS stack so that they'll be
    # added to children if needed.
    #
    nsframe= self.retrieveNamespaces(node)
    if len(nsframe)>0:
      self._nsstack.append(nsframe)

    # Compile the subtemplate with reset variable counters. We can start again
    # at __pxtl_tagname_0 (etc.) because these will all be local variables,
    # not clashing with the global/nested vars.
    #
    oldNs= (
      self._n_tagname, self._n_elcond, self._n_atcond, self._n_index,
      self._n_min, self._haveElcond,
    )
    self._n_tagname=self._n_elcond=self._n_atcond=self._n_index=self._n_min= 0
    self._haveElcond= False
    self.compileChildren(node)

    self._write(dent= -1)
    if funcname=='__pxtl_v':
      self.assignTo(fn)
    (
      self._n_tagname, self._n_elcond, self._n_atcond, self._n_index,
      self._n_min, self._haveElcond
    )= oldNs

    if len(nsframe)>0:
      del self._nsstack[-1]


  def callElement(self, node):
    args= node.getAttribute('args')
    fn= node.getAttribute('fn')
    if fn=='':
      raise PXTLAttributeMissingError(node, 'fn')

    # Store the namespace context for the subtemplate to pick up then call it
    #
    self.storeNamespaces(node)
    self._write('%s(%s)' % (fn, args), node)


  def importElement(self, node):
    """ Import an external resource. Can be a PXTL template to be transformed
        or a simple text or XML resource.
    """
    # Get absolute URI from src attribute and template's original URI.
    #
    src= node.getAttributeNode('src')
    if src is None:
      raise PXTLAttributeMissingError(node, 'src')
    self.assignFrom(src, var= 'src')
    self._write('if __pxtl_baseuri is not __pxtl.constants.none:', dent=1)
    self._write('__pxtl_src= __pxtl.constants.urlparse.urljoin(__pxtl_baseuri,'
      '__pxtl_src)', dent= -1
    )

    # Get content type of specified resource. If not given in a type attribute
    # ask urllib to guess a type (will generally use file extension).
    #
    typeNode= node.getAttributeNode('type')
    typeStatic= False
    if typeNode is not None:
      typeStatic, typeValue= evalStatic(typeNode.value)
      if typeStatic:
        typeValue= string.lower(typeValue)
      else:
        self.assignFrom(typeNode)
    else:
      self._write(
        'if __pxtl.constants.urlparse.urlparse(__pxtl_src)[0]=="file":', dent= 1
      )
      self._write(
        '__pxtl_v= __pxtl.constants.mimetypes.guess_type(__pxtl_src)[0]',
      node, dent= -1)
      self._write('else:', dent= 1)
      self._write(
        '__pxtl_f= __pxtl.utilities.importRead(__pxtl_src, __pxtl_object)',
      node)
      self._write('__pxtl_v= __pxtl_f.info().gettype()')
      self._write('__pxtl_f.close()', dent= -1)
      self._write('__pxtl_v= __pxtl.constants.string.lower(__pxtl_v or "")')

    # XML types -> include literally, without altering markup.
    #
    if not typeStatic:
      self._write('if __pxtl_v in __pxtl.constants.TYPES_XML:', dent= 1)
    if not typeStatic or typeValue in TYPES_XML:
      self._write(
        '__pxtl_f=__pxtl.utilities.importRead(__pxtl_src,__pxtl_object)',node
      )
      self._write('__pxtl_writer.write(__pxtl_f.read())')
      self._write('__pxtl_f.close()')
    if not typeStatic:
      self._write(dent= -1)

    # Text types -> include with markup-encoding.
    #
    if not typeStatic:
      self._write('elif __pxtl_v in __pxtl.constants.TYPES_TEXT:', dent= 1)
    if not typeStatic or typeValue in TYPES_TEXT:
      self._write(
        '__pxtl_f=__pxtl.utilities.importRead(__pxtl_src,__pxtl_object)',node
      )
      self._write('__pxtl_serialiser.writeString(__pxtl_f.read())')
      self._write('__pxtl_f.close()')
    if not typeStatic:
      self._write(dent= -1)

    # Otherwise, treat it as a PXTL template. See if there is a 'globals' attr
    # set, if not make a fresh new global scope for the imported template.
    # See if there is an 'as' attr set, if so store the globals for posterity.
    #
    if not typeStatic:
      self._write('else:', dent= 1)
    if not typeStatic or (
      typeValue not in TYPES_XML and typeValue not in TYPES_TEXT
    ):
      globalsNode= node.getAttributeNode('globals')
      if globalsNode is not None:
        self.assignFrom(globalsNode)
      else:
        self._write('__pxtl_v= {}')
      asNode= node.getAttributeNode('as')

      # Have the runtime call the optimised implementation back to process the
      # imported template. Make sure the template knows that its output method
      # must match the caller's, and ensure it has the same global context
      # object (used for communicating namespaces and errors between different
      # templates).
      #
      self._write('__pxtl_v["__pxtl_object"]= __pxtl_object')
      self._write('__pxtl_src= __pxtl.constants.urlparse.urlparse(__pxtl_src)')
      self._write('if __pxtl_src[0]=="file":', dent= 1)
      self._write(
        '__pxtl_src= __pxtl.constants.urllib.url2pathname(__pxtl_src[2])',dent=-1
      )
      self._write('else:', dent= 1)
      self._write(
        'raise __pxtl.exceptions.PXTLImportSrcError(__pxtl_object,__pxtl_src)',
        src, dent= -1
      )
      self.storeNamespaces(node)
      self._write(
        '__pxtl.optimised.processFile(__pxtl_src, __pxtl_writer, '
        '__pxtl_v, 0, 0, __pxtl_dom, __pxtl_bytecodebase, %s, %s)' %
        (repr(self.method), repr(self.encoding)),
      )
      if asNode is not None:
        self._write('__pxtl_v= __pxtl.utilities.DictAsInstance(__pxtl_v)')
        self.assignTo(asNode)
    if not typeStatic:
      self._write(dent=-1)


  # Store all namespaces currently in scope in the run-time global context,
  # so that they can be read back by called code. Used by callElement, might
  # also be a good idea to make importElement use it?
  #
  def storeNamespaces(self, node):
    nss= getNamespaces(node.parentNode)
    namespaces= {}
    maybenamespaces= {}
    for prefix, namespaceURI in nss.items():
      if namespaceURI!=PXNS:
        if prefix is None:
          name= 'xmlns'
        else:
          name= 'xmlns:'+prefix

        # For each in-scope namespace, check the current nsstack frame to see
        # if it's actually-not-in-scope (due to skipped elements) or
        # only-maybe-in-scope (due to conditional elements and call-time
        # namespaces).
        #
        for attr, levels, callns in self._nsstack[-1]:
          if name==attr.name:
            if levels!=() or callns:
              maybenamespaces[name]= (namespaceURI, levels, callns)
            break
        else:
          namespaces[name]= namespaceURI
    self._write('__pxtl_object.callns= %s' % repr(namespaces))

    # For the namespaces that were found to be maybe-in-scope, do run-time
    # checks to see whether they actually were; if so, add them to the context
    # global too.
    #
    for name, (namespaceURI, levels, callns) in maybenamespaces.items():
      condlist= map(lambda level: '__pxtl_atcond_%d.old_success' % level, levels)
      if callns:
        condlist[:0]= ['%s not in __pxtl_callns' % repr(name)]
      self._write('if %s:' % string.join(condlist, ' or '), dent= 1)
      self._write('__pxtl_object.callns[%s]= %s' %
        (repr(name), repr(namespaceURI)), dent= -1
      )

  # On the called subtemplate, grab the NS details back out of the
  # context global and find any declarations we still need to make. Return the
  # callns frame for the compiler.
  #
  def retrieveNamespaces(self, node):
    nsframe= []
    nss= getNamespaces(node)
    self._write('__pxtl_callns= []')
    for prefix, namespaceURI in nss.items():
      if namespaceURI!=PXNS:
        if prefix is None:
          name= 'xmlns'
        else:
          name= 'xmlns:'+prefix
        self._write('if __pxtl_object.callns.get(%s)!=%s:' %
          (repr(name), repr(namespaceURI)), dent= 1
        )
        self._write('__pxtl_callns.append(%s)' % repr(name), dent= -1)
        nsattr= node.ownerDocument.createAttributeNS(NSNS, name)
        nsattr.value= namespaceURI
        nsframe.append( (nsattr, (), True ) )
    return nsframe


# PXTLObject - implement the in-scope 'pxtl' object, redirecting anything sent
# to write() to the output stream. Also simulate enough of the DOM Node
# interface to make PXTLError work with it as a pseudo-Node source of errors
# for run time where there is no real DOM Node available.
#
class PXTLObject(AbstractPXTLObject):
  inblock= False
  serialiser= None

  # Hack: catch the case of template code from incompatible old beta versions
  # of pxtl.optimised being used. These don't have the proper version-number
  # checking, so catch them by their attempt to instantiate PXTLObject without
  # a parameter.
  #
  def __init__(self, beta= True):
    if beta:
      raise OutputMismatch()

    self.pxdomLocation= self
    self.ownerDocument= self
    self.parentNode= None
    self.ownerElement= None
    self.context(-1, -1, '', 0, '')
    self.callns= []

  def write(self, text, coding='text'):
    if self.inblock is None or self.serialiser is None:
      raise StandardError('pxtl.write can only be called in code PIs')
    if not isinstance(text, StringTypes):
      text= str(text)
    if coding=='mark':
      self.serialiser.w(text)
    elif coding!='text':
      self.serialiser.w(encodeText(text, coding))
    else:
      self.serialiser.writeString(text)

  def context(self, columnNumber, lineNumber, uri, nodeType,nodeName):
    self.columnNumber= columnNumber
    self.lineNumber= lineNumber
    self.uri= uri
    self.documentURI= uri
    self.nodeType= nodeType
    self.nodeName= nodeName
  doctype= None
  [ELEMENT_NODE,ATTRIBUTE_NODE,TEXT_NODE,CDATA_SECTION_NODE,
  ENTITY_REFERENCE_NODE,ENTITY_NODE,PROCESSING_INSTRUCTION_NODE,COMMENT_NODE,
  DOCUMENT_NODE,DOCUMENT_TYPE_NODE,DOCUMENT_FRAGMENT_NODE,NOTATION_NODE
  ]= range(1,13)
