""" Reference implementation of the Python XML Templating Language.
"""

__all__    = ['processFile', 'Transformer', 'PXTLObject']

import os, sys, types, operator, string, urllib, urlparse
from pxtl.constants import *
from pxtl.exceptions import *
from pxtl.serialisers import *
from pxtl.utilities import *

def processFile(
  path, writer= None, globals= None, debug= False, headers= False, dom= None,
  bytecodebase= None
):
  """ Read, deserialise and transform a template and any subtemplates it
      imports, from a file in the filesystem. Optional arguments:

      writer  - stream-like object to send serialised output (default stdout)
      globals - dictionary to use as template global scope (default {})
      debug   - output a debugging page if there's an error (default False)
      headers - output HTTP response headers (for CGI scripts, default False)
      dom     - DOM L3 Core/XML/Load implementation to use (default pxdom)
  """
  if writer is None:
    writer= sys.stdout
  if dom is None:
    from pxtl import pxdom
    dom= pxdom.getDOMImplementation('')
  try:
    parser= dom.createLSParser(dom.MODE_SYNCHRONOUS, None)
  except NameError:
    raise PXTLNoParserError(None)
  parser.domConfig.setParameter('entities', True)
  parser.domConfig.setParameter('cdata-sections', True)
  if parser.domConfig.canSetParameter('pxdom-resolve-resources', False):
    parser.domConfig.setParameter('pxdom-resolve-resources', False)
  t= Transformer()
  uri= 'file:'+urllib.pathname2url(os.path.abspath(path))
  try:
    try:
      doc= parser.parseURI(uri)
    except NonErrors:
      raise
    except Exception, e:
      raise PXTLParseError(e)
    doc= t.transformDocument(doc, globals)
    if not SERIALISERS.has_key(t.method):
      raise PXTLMethodValueError(t.method)
  except PXTLError, e:
    if not debug:
      raise
    if headers:
      writer.write('Status: 500 Server error\r\n')
    processFile(ERROR_PAGE,writer,{'e':e,'partial':None},False,headers,dom)
    return
  encoding= None
  if UnicodeType is not None:
    try:
      encoding= doc.inputEncoding
    except AttributeError:
      pass
    if encoding is None:
      try:
        encoding= doc.xmlEncoding
      except AttributeError:
        pass
    if encoding is None:
      encoding= 'utf-8'
  if headers:
    writeTypeHeader(writer, t.mimetype, t.method, encoding)
  ser= SERIALISERS[t.method](writer, encoding)
  ser.writeDocument(doc, t.literals)


class Transformer:
  """ Effect PXTL transformations on a DOM Level 2 Core-compliant DOM tree.
  """
  def __init__(self):
    self.pxtl= PXTLObject()

  def transformDocument(self, document, globals= None):
    """ Main public interface. PXTL-transform an XML Document node, executing
        Python code with the optional dictionary 'globals' as global scope.
        Returns the transformed Document node, which may or may not be the
        original node, depending on DOM. Places the template's desired output
        method in the Transformer's 'method' property and a list of
        pre-serialised text nodes (from px_mark PI etc) in the property
        'literals', which the caller may read if interested.
    """
    root= document.documentElement
    dom= document.implementation

    # Try to make a Level 3 LSParser object so we can load new documents. If
    # this is not doable, px:import elements will not work.
    #
    self.parser= None
    try:
      self.parser= dom.createLSParser(dom.MODE_SYNCHRONOUS, None)
      self.parser.domConfig.setParameter('entities', True)
      self.parser.domConfig.setParameter('cdata-sections', True)
    except AttributeError:
      pass

    # Make a new variable-environment-holding scoper object. If a global scope
    # was supplied, store it in the scoper. If the nested_scope feature is on
    # (either because it is always on in this version of Python, or the user
    # asked for it in the future attribute), ask the scoper to use nesteds.
    #
    if globals is None:
      globals= {}
    globals['pxtl']= self.pxtl
    locals= None
    features= checkFeatures(root.getAttributeNodeNS(PXNS, 'future'))
    if haveNestedScopes(features):
      nesteds= []
    else:
      nesteds= None
    self.scoper= Scoper(globals, locals, nesteds, features)

    # Make a copy of the locals builtin function, in case the template
    # overwrites it with something else.
    #
    globals['__pxtl_locals']= self.scoper.pxeval('locals', None)

    # Work out what doctype and output method the output document will be,
    # from the PXTL doctype attribute on the root.
    #
    doctype= root.getAttributeNodeNS(PXNS, 'doctype')
    if doctype is not None:
      doctuple= self.scoper.pxeval(doctype.value, doctype)
      if type(doctuple)!=types.TupleType:
        raise PXTLDocumentTypeError(doctype, doctuple)
      doctuple= (doctuple+(None,)*5)[:5]
    else:
      doctuple= (None,)*5
    self.mimetype, self.method, publicId, systemId, internalSubset= doctuple
    self.method= determineMethod(doctuple, root.namespaceURI, root.localName)

    # Run transformations on the document and return the document shorn of any
    # reference to now-unused PXTL namespaces.
    #
    self.literals= []
    self.elementContext= ConditionalContext()
    self.attributeContext= ConditionalContext()
    self.whitespace= True
    self.transformChildren(document)
    removePXTLNamespaces(document)

    # Write the previously-calculated doctype (or lack of doctype) to the
    # document; this may cause the document to be cloned.
    #
    if doctype is not None and (
    publicId is not None or systemId is not None or internalSubset is not None
    ):
      doctype= dom.createDocumentType(
        document.documentElement.tagName, publicId, systemId
      )
    else:
      doctype= None
    document= setDoctype(document, doctype)
    return document


  # Transformation of DOM Nodes.
  #
  def transformChildren(self, node, rslice= None):
    """ Apply PXTL transformations to all child nodes, or to the last 'rslice'
        child nodes if supplied.
    """
    # Despatch each transformable child to relevant function.
    #
    children= list(node.childNodes)
    if rslice is not None:
      children[:len(children)-rslice]= []
    elementContext= self.elementContext
    attributeContext= self.attributeContext
    self.elementContext= ConditionalContext()
    for child in children:
      if child.nodeType==node.ELEMENT_NODE:
        self.attributeContext= attributeContext.copy()
        whitespace= self.whitespace
        self.transformElement(child)
        self.whitespace= whitespace
      elif child.nodeType==node.PROCESSING_INSTRUCTION_NODE:
        self.transformPI(child)
    self.elementContext= elementContext
    self.attributeContext= attributeContext


  def transformElement(self, node):
    """ Perform all PXTL transformations on an element.
    """
    # Ignore/remove doctype attribute (only used by processDocument function
    # in root element of root template), and future attribute (only used in
    # root element of any template).
    #
    attr= node.getAttributeNodeNS(PXNS, 'doctype')
    if attr is not None:
      node.removeAttributeNode(attr)
    attr= node.getAttributeNodeNS(PXNS, 'future')
    if attr is not None:
      node.removeAttributeNode(attr)

    # If there is a PXTL space attribute on the element, change whitespace
    # processing mode.
    #
    attr= node.getAttributeNodeNS(PXNS, 'space')
    if attr is not None:
      whitespace= self.scoper.pxeval(attr.value, attr)
      if whitespace is not None:
        self.whitespace= [True, False][not whitespace]
      node.removeAttributeNode(attr)

    # If we are in no-whitespace mode, remove all whitespace in child element
    # content not surrounded by text. This assumes the document has been read
    # in a normal form, ie. there is only one Text node between non-Text
    # nodes.
    #
    if not self.whitespace:
      for child in list(node.childNodes):
        if child.nodeType==child.TEXT_NODE:
          data= string.strip(child.data)
          if data=='':
            node.removeChild(child)
          else:
            child.data= data

    # If this is a none-element and there's no tagname attribute to rename it,
    # it is unsuccessful. Otherwise, look for conditional attributes to
    # determine success.
    #
    if (
      node.namespaceURI==PXNS and node.localName=='none' and
      not node.hasAttributeNS(PXNS, 'tagname')
    ):
      success= False
    else:
      success= None
      attrs= attrList(node.attributes)
      for attr in attrs:
        cond= attr.localName
        if attr.namespaceURI==PXNS and cond in CONDITIONALS:
          if success is not None:
            raise PXTLMultipleConditionalError(attr.ownerElement)
          if self.attributeContext.getTesting(cond, attr):
            testvalue= operator.truth(self.scoper.pxeval(attr.value, attr))
          else:
            testvalue= None
          success= self.attributeContext.getSuccess(cond, testvalue)
          node.removeAttributeNode(attr)

    # Remove unsuccessful elements.
    #
    if success==False:
      self.transformChildren(node)
      removeReparent(node)
      return

    # Change tag name if tagname attribute present.
    #
    attr= node.getAttributeNodeNS(PXNS, 'tagname')
    if attr is not None:
      qname= self.scoper.pxeval(attr.value, attr)
      if not isinstance(qname, StringTypes) or qname=='':
        raise PXTLTagnameAttrError(attr, qname)

      # Work out what namespace the element with a possibly-changed prefix
      # would be in.
      #
      if string.count(qname, ':')==1 and ':' not in qname[0]+qname[-1]:
        ns= node.lookupNamespaceURI(string.split(qname, ':')[0])
      else:
        ns= node.lookupNamespaceURI(None)
      if ns in [PXNS, NSNS] or qname=='xmlns':
        raise PXTLTagnamespaceAttrError(attr, ns)

      # Make a new element with the desired tagname, then move all attributes
      # and children from the old node to the new one. (Because attributes
      # are immutable; we could do this with DOM Level 3 renameNode, but
      # fixdom imps don't currently support this.)
      #
      if ':' in qname and ns is None:
        newel= node.ownerDocument.createElement(qname)
      else:
        newel= node.ownerDocument.createElementNS(ns, qname)
      for attr in attrList(node.attributes):
        if not (attr.namespaceURI==PXNS and attr.localName=='tagname'):
          newel.setAttributeNodeNS(attr.cloneNode(True))
      for child in list(node.childNodes):
        newel.appendChild(child)
      node.parentNode.replaceChild(newel, node)
      node= newel

    # Check all attributes for pseudo-PIs.
    #
    for attr in attrList(node.attributes):
      if string.find(attr.value, '{?')!=-1:
        parts= parsePseudoPIs(attr.value, attr)

        # If there are any px_if pseudo-PIs that evaluate False, throw the
        # attribute away before evaluating anything else.
        #
        success= True
        for part in parts:
          if type(part)==types.TupleType and part[0]=='px_if':
            if not self.scoper.pxeval(part[1], attr):
              success= False
              break
        if not success:
          node.removeAttributeNode(attr)
          continue

        # Replace all other pseudo-PIs with their values.
        #
        for i in range(len(parts)):
          if type(parts[i])==types.TupleType:
            (target, data)= parts[i]
            if target=='px_if':
              parts[i]= ''
              continue
            elif SHORTCUTS.has_key(target):
              target= SHORTCUTS[target]
            elif target[:3]=='px_':
              target= target[3:]
            else:
              raise PXTLPseudoPITargetPXError(attr, target)
            if target=='note':
              parts[i]= ''
            elif target in CODINGS:
              d= self.scoper.pxeval(data, attr)
              if not isinstance(d, StringTypes):
                d= str(d)
              parts[i]= encodeText(d, target)
            else:
              raise PXTLPseudoPITargetNameError(attr, target)

        # Write the finished attribute back to the node.
        #
        attr.value= string.join(parts, '')

    # If there is a PXTL attr attribute, a dictionary, evaluate it and send
    # all its values to new/replacement attributes on the element.
    #
    attr= node.getAttributeNodeNS(PXNS, 'attr')
    if attr is not None:
      dict= self.scoper.pxeval(attr.value, attr)
      if type(dict)!=types.DictionaryType:
        raise PXTLAttrTypeError(attr, dict)
      for (qname, value) in dict.items():
        if not isinstance(qname, StringTypes):
          raise PXTLAttrNameTypeError(attr, qname)
        node.removeAttribute(qname)
        if value is not None:
          if not isinstance(value, StringTypes):
            value= str(value)
          if string.count(qname, ':')==1 and ':' not in qname[1]+qname[-1]:
            ns= node.lookupNamespaceURI(string.split(qname, ':')[0])
          else:
            ns= node.lookupNamespaceURI(None)
          newattr= node.ownerDocument.createAttributeNS(ns, qname)
          newattr.value= value
          node.setAttributeNodeNS(newattr)
      node.removeAttributeNode(attr)

    # Handle PXTL elements.
    #
    if node.namespaceURI==PXNS and node.localName in CONDITIONALS:
      self.conditionalElement(node)
    else:
      self.elementContext.reset()
      if node.namespaceURI==PXNS:
        if node.localName=='for':
          self.forElement(node)
        elif node.localName=='while':
          self.whileElement(node)
        elif node.localName=='def':
          self.defElement(node)
        elif node.localName=='call':
          self.callElement(node)
        elif node.localName=='import':
          self.importElement(node)
        else:
          raise PXTLTagNameError(node)
      else:
        self.transformChildren(node)


  def transformPI(self, node):
    """ Perform PXTL transformations on a Processing Instruction.
    """
    # Only interested in PIs with target name px_..., or shortcuts for them.
    # Complain about px_... targets we do not recognise.
    #
    if SHORTCUTS.has_key(node.target):
      target= SHORTCUTS[node.target]
    elif node.target[:3]=='px_':
      target= node.target[3:]
    elif node.target[:3]=='px:':
      raise PXTLPITargetNameError(node)
    else:
      return
    if target not in TARGETS:
      raise PXTLPITargetNameError(node)

    # Just delete comment PIs.
    #
    if target=='note':
      node.parentNode.removeChild(node)

    # Execute PIs that produce output. If not mark-up, encode the output.
    # Replace the node with any output produced by the PI.
    #
    elif target in CODINGS+['mark']:
      d= self.scoper.pxeval(node.data, node)
      if not isinstance(d, StringTypes):
        d= str(d)
      if target!='mark':
        d= encodeText(d, target)
      if d=='':
        node.parentNode.removeChild(node)
      else:
        output= node.ownerDocument.createTextNode(d)
        if target=='mark':
          self.literals.append(output)
        node.parentNode.replaceChild(output, node)

    # Execute the code PI. Generates no markup directly, but may call write()
    # on the pxtl object to output literal and/or encoded output. Check the
    # pxtl object after executing the code and grabs anything that has been
    # output, to replace the original node.
    #
    elif target=='code':
      self.pxtl._begin()
      if node.data[:2]==':\n':
        self.scoper.pxblock(node.data[2:], node)
      else:
        self.scoper.pxexec(string.strip(node.data), node)
      texts= self.pxtl._read()
      for (literal, text) in texts:
        output= node.ownerDocument.createTextNode(text)
        if literal:
          self.literals.append(output)
        node.parentNode.insertBefore(output, node)
      node.parentNode.removeChild(node)
    else:
      raise PXTLPITargetNameError(node)


  # Specific PXTL elements
  #
  def conditionalElement(self, node):
    """ Simple conditional element. Remove it from the document if test False.
    """
    test= None
    cond= node.localName
    if cond!='else':
      test= node.getAttributeNode('test')
      if test is None:
        raise PXTLAttributeMissingError(node, 'test')

    if self.elementContext.getTesting(cond, node):
      testvalue= operator.truth(self.scoper.pxeval(test.value, test))
    else:
      testvalue= None
    success= self.elementContext.getSuccess(cond, testvalue)

    if success:
      self.transformChildren(node)
      removeReparent(node)
    else:
      node.parentNode.removeChild(node)


  def forElement(self, node):
    """ Include and transform child nodes once for each item in a list,
        assigning loop variable each time around.
    """
    # Move loop body into a holding fragment.
    #
    body= node.ownerDocument.createDocumentFragment()
    for child in list(node.childNodes):
      body.appendChild(child)
    children= list(body.childNodes)

    # Get iterable object from in or range attribute.
    #
    rattr= node.getAttributeNode('range')
    iattr= node.getAttributeNode('in')
    if (rattr is None) == (iattr is None):
      raise PXTLForItemsError(node)
    if iattr is not None:
      items= self.scoper.pxeval(iattr.value, iattr)
    else:
      r= self.scoper.pxeval(rattr.value, rattr)
      if type(r) in (types.IntType, types.LongType):
        r= (r,)
      if type(r)!=types.TupleType:
        raise PXTLForRangeError(rattr, r)
      items= apply(xrange, r)

    # Iterate over items, optionally assigning iteration item and index vars.
    #
    item= node.getAttributeNode('item')
    index= node.getAttributeNode('index')
    success= len(items)>0
    ix= 0
    while ix<len(items):
      if item is not None:
        self.scoper.pxassign(item.value, items[ix], item)
      if index is not None:
        self.scoper.pxassign(index.value, ix, item)

      # Each time round the loop, make a copy of the body inside the loop and
      # transform it.
      #
      for child in children:
        node.appendChild(child.cloneNode(True))
      self.transformChildren(node, rslice= len(children))
      ix= ix+1

    # Finished, remove the template's for element leaving the iteration
    # results at this level.
    #
    self.elementContext.getSuccess('for', success)
    removeReparent(node)


  def whileElement(self, node):
    """ Repeatly include and transform child elements.
    """
    # Get test attribute and optional min, index.
    #
    test= node.getAttributeNode('test')
    if test is None:
      raise PXTLAttributeMissingError(node, 'test')
    attr= node.getAttributeNode('min')
    if attr is None:
      min= 0
    else:
      min= self.scoper.pxeval(attr.value, attr)
      if not isinstance(min, IntTypes):
        raise PXTLWhileMinError(attr, min)
    index= node.getAttributeNode('index')

    # Move loop body into a holding fragment.
    #
    body= node.ownerDocument.createDocumentFragment()
    for child in list(node.childNodes):
      body.appendChild(child)
    children= list(body.childNodes)

    # Continue until test condition evaluates False. Do not test the condition
    # for the first min times around the loop. Assign loop index if index
    # attribute used.
    #
    i= 0
    success= False
    while True:
      if i>=min:
        if not self.scoper.pxeval(test.value, test):
          break
      success= True
      if index is not None:
        self.scoper.pxassign(index.value, i, index)
      i= i+1

      # Make a copy of the loop body in the loop element and apply PXTL
      # transformations.
      #
      for child in children:
        node.appendChild(child.cloneNode(True))
      self.transformChildren(node, rslice= len(children))

    # Finished, remove loop element.
    #
    self.elementContext.getSuccess('while', success)
    removeReparent(node)


  def defElement(self, node):
    """ Define a subtemplate that can be called like a function.
    """
    if not node.hasAttribute('fn'):
      raise PXTLAttributeMissingError(node, 'fn')
    fn= node.getAttribute('fn')
    args= None
    if node.hasAttribute('args'):
      args= node.getAttribute('args')

    # Move subtemplate definition into a holding fragment.
    #
    body= node.ownerDocument.createDocumentFragment()
    for child in list(node.childNodes):
      body.appendChild(child)

    # Make the subtemplate object which will simulate function calls. Store
    # the currently active namespaces in it so they are restored when the
    # subtemplate is called in a different part of the document.
    #
    t= Subtemplate(
      body, args, getNamespaces(node), self.scoper, node, self.whitespace
    )
    t.documentURI= node.ownerDocument.documentURI

    # Put the subtemplate in scope and remove its definition node.
    #
    self.scoper.pxassign(fn, t, node)
    node.parentNode.removeChild(node)


  def callElement(self, node):
    """ Call a previously-defined subtemplate.
    """
    # Get the subtemplate object referred to.
    #
    fn= node.getAttributeNode('fn')
    if fn is None:
      raise PXTLAttributeMissingError(node, 'fn')
    fn= self.scoper.pxeval(fn.value, fn)
    if not isinstance(fn, Subtemplate):
      raise PXTLCallTypeError(node, fn)

    # Remove namespaces from this element, as generated children from the def
    # elements should not inherit the call's own namespaces.
    #
    for attr in attrList(node.attributes):
      if attr.namespaceURI==NSNS:
        node.removeAttributeNode(attr)

    # Restore any namespaces that were defined on the def element but are
    # different or undefined now.
    #
    nss= getNamespaces(node.parentNode)
    for (prefix, uri) in fn.nss.items():
      if not nss.has_key(prefix) or nss[prefix]!=uri:
        qname= 'xmlns'
        if prefix is not None:
          qname= 'xmlns:'+prefix
        node.setAttributeNS(NSNS, qname, uri)

    # Assign the argument list to the appropriate variables in a new nested
    # scope.
    #
    args= None
    if node.hasAttribute('args'):
      args= node.getAttribute('args')
    parentScoper= self.scoper
    parentDocumentURI= node.ownerDocument.documentURI
    parentSpace= self.whitespace
    self.scoper= fn.scoperWithArgs(args, parentScoper, node)
    node.ownerDocument.documentURI= fn.documentURI
    self.whitespace= fn.whitespace

    # Copy the holding fragment back into the document at the point of the
    # call element. Apply PXTL transformations using the new nested scope and
    # remove the call element.
    #
    for child in fn.body.childNodes:
      node.appendChild(child.cloneNode(True))
    self.transformChildren(node)
    node.ownerDocument.documentURI= parentDocumentURI
    self.scoper= parentScoper
    self.whitespace= parentSpace
    removeReparent(node)


  def importElement(self, node):
    """ Import element handler. Replace import element with either text from
        the imported resource, or the DOM of a PXTL template, depending on the
        type of the imported resource.
    """
    if self.parser is None:
      raise PXTLNoParserError(node)
    src= node.getAttributeNode('src')
    if src is None:
      raise PXTLAttributeMissingError(node, 'src')
    src= self.scoper.pxeval(src.value, src)
    if node.ownerDocument.documentURI is not None:
      src= urlparse.urljoin(node.ownerDocument.documentURI, src)

    # If type attribute specified, decide how to include based on that,
    # otherwise try to work out what type the resource is by asking mimetypes
    # to guess (if it's a file) or urllib to open it (if it's something else -
    # urllib can't be used for files as it misleadingly reports unknown file
    # types are text/plain).
    #
    t= node.getAttributeNode('type')
    if t is not None:
      t= self.scoper.pxeval(t.value, t)
    elif urlparse.urlparse(src)[0]=='file':
      t= mimetypes.guess_type(src)[0]
      if t is not None:
        t= string.lower(t)
    else:
      stream= importRead(src, node)
      t= string.lower(stream.info().gettype())
      stream.close()

    # For non-PXTL includes, read the resource and include it as text. If it's
    # an XML document already, include it as literal pre-serialised content.
    #
    if t in TYPES_TEXT+TYPES_XML:
      stream= importRead(src, node)
      content= stream.read()
      stream.close()
      text= node.ownerDocument.createTextNode(content)
      node.parentNode.replaceChild(text, node)
      if t in TYPES_XML:
        self.literals.append(text)

    # For PXTL includes, get the resource as a DOM Document and copy its
    # transformed content to the current document.
    #
    else:
      try:
        idoc= self.parser.parseURI(src)
      except IOError:
        raise PXTLImportSrcError(node, src)
      except NonErrors:
        raise
      except Exception, e:
        raise PXTLParseError(e)
      iroot= idoc.documentElement

      # Get the new global scope to be used in this file from the globals
      # attribute, or use a new blank dictionary if none is supplied.
      #
      globalsAttr= node.getAttributeNode('globals')
      if globalsAttr is None:
        globalsDict= {}
      else:
        globalsDict= self.scoper.pxeval(globalsAttr.value, globalsAttr)
        if type(globalsDict)!=types.DictType:
          raise PXTLImportGlobalsError(globalsAttr, globalsDict)
      globalsDict['pxtl']= self.pxtl

      # Get features of imported document and make a new Scoper to hold them
      # and the global scope.
      #
      features= checkFeatures(iroot.getAttributeNodeNS(PXNS, 'future'))
      if haveNestedScopes(features):
        nesteds= []
      else:
        nesteds= None
      nestedScoper= Scoper(globalsDict, None, nesteds, features)

      # If 'as' attribute passed, write what will be the global scope of the
      # imported file to the given l-value in the importing scope.
      #
      as= node.getAttributeNode('as')
      if as is not None:
        self.scoper.pxassign(as.value, DictAsInstance(globalsDict), as)

      # Import each root-level node of the document into the import element.
      # Add single newline text nodes between each document-level node.
      #
      while node.firstChild is not None:
        node.removeChild(node.firstChild)
      for child in list(idoc.childNodes):
        if node.childNodes.length!=0:
          node.appendChild(node.ownerDocument.createTextNode('\n'))
        if child.nodeType!=child.DOCUMENT_TYPE_NODE:
          node.appendChild(node.ownerDocument.importNode(child, True))

      # Apply PXTL transformations and put the results into the importing
      # document.
      #
      _scoper= self.scoper
      _whitespace= self.whitespace
      _elementContext= self.elementContext
      _attributeContext= self.elementContext
      _documentURI= node.ownerDocument.documentURI
      self.scoper= nestedScoper
      self.whitespace= True
      self.elementContext= ConditionalContext()
      self.attributeContext= ConditionalContext()
      node.ownerDocument.documentURI= src

      self.transformChildren(node)
      removeReparent(node)

      self.scoper= _scoper
      self.whitespace= _whitespace
      self.elementContext= _elementContext
      self.attributeContext= _attributeContext
      node.ownerDocument.documentURI= _documentURI


# Reference implementation utility objects
#
class PXTLObject(AbstractPXTLObject):
  def write(self, text, coding='text'):
    """ Can only be called in a px_code PI, otherwise pxtlobject is not
        writeable. Adds possibly-encoded text to a list of text nodes to
        create. For 'mark' encoding the output will end up in a literal
        node.
    """
    if not self.writeable:
      raise Exception('pxtl.write can only be called in code PIs')
    if not isinstance(text, StringTypes):
      text= str(text)
    literal= coding=='mark'
    if not literal:
      text= encodeText(text, coding)
    if len(self.texts)>0 and self.texts[-1][0]==literal:
      self.texts[-1]= (literal, self.texts[-1][1]+text)
    else:
      self.texts.append((literal, text))

  def __init__(self):
    self.writeable= False

  def _begin(self):
    self.writeable= True
    self.texts= []

  def _read(self):
    self.writeable= False
    return self.texts


class Subtemplate:
  """ Class representing a subtemplate function. Used to interpret parameter
      lists.
  """
  def __init__(self, body, args, nss, scoper, context, whitespace):
    self.body= body
    self.nss= nss
    self.scoper= scoper
    self.args= args
    if args is not None:
      code= 'def __pxtl_assign(%s): return __pxtl_locals()' % args
      scoper.pxexec(code, context)
      locals= scoper.getLocals()
      self.readargs= locals['__pxtl_assign']
      del locals['__pxtl_assign']
    self.whitespace= whitespace

  def scoperWithArgs(self, args, scoper, context):
    if args is not None and self.args is None:
      raise PXTLCallArgsError(context, args)
    if self.args is None:
      l= {}
    else:
      locals= scoper.getLocals()
      locals['__pxtl_assign']= self.readargs
      code= '__pxtl_assign= __pxtl_assign(%s)' % (args or '')
      scoper.pxexec(code, context)
      l= locals['__pxtl_assign']
      del locals['__pxtl_assign']
    return self.scoper.makeNestedScoper(l)


class Scoper:
  """ Represents a code block's variable environment. Includes a local scope,
      global scope, and in Python 2.2 and later (or 2.1 with a suitable future
      declaration) a set of nested scopes. Also holds, but does not do
      anything with, a list of future-features.
  """
  def __init__(self, globals, locals, nesteds, features):
    """ Construct a new scoper with optional local scope and a (possibly
        empty) list of nested scopes. If nesteds is 'None' instead of a list,
        nested scopes are disabled.
    """
    self.globals= globals
    self.locals= locals
    self.nesteds= nesteds
    self.features= features

  def makeNestedScoper(self, locals):
    """ Return a scoper that will emulate the variable environment of a
        function nested inside the current environment. The new local scope is
        supplied as a parameter. Global scope is the same as the current
        scoper's. If nested scopes are enabled the old local scope is added to
        list of nested scopes in the new scoper.
    """
    if self.nesteds is None:
      return Scoper(self.globals, locals, None, self.features)
    if self.locals is None:
      return Scoper(self.globals, locals, [], self.features)
    nesteds= self.nesteds+[self.locals]
    return Scoper(self.globals, locals, nesteds, self.features)

  def getLocals(self):
    """ Return the locals dictionary, or globals if there is none.
    """
    if self.locals is not None:
      return self.locals
    else:
      return self.globals

  def getGlobals(self):
    """ Return a globals dictionary with any nested scopes combined into it.
    """
    if self.nesteds is None or len(self.nesteds)==0:
      return self.globals
    globals= self.globals.copy()
    for nested in self.nesteds:
      globals.update(nested)
    return globals

  # Python wrappers. Run statements, expressions and assignments in an
  # environment described by the current scoper object.
  #
  def _pxcode(self, code, context):
    """ Base code wrapper, used by pxexec, pxeval, pxassign, pxblock. Run some
        wrapped code in the current scope. Catch any error and re-throw as a
        PXTLRuntimeError; if the context node is a pxdom node, bodge the line
        numbering so that wrapped exceptions appear to come from the context
        node's file position.
    """
    # Work out source position of the context node.
    #
    lineNumber= filename= uri= None
    if context is not None:
      if hasattr(context, 'pxdomLocation'):
        uri= context.pxdomLocation.uri
        lineNumber= context.pxdomLocation.lineNumber
        if lineNumber==-1:
          lineNumber= None
      else:
        uri= context.ownerDocument.documentURI
    if uri is not None:
      uri= urlparse.urlparse(uri)
      if uri[0]=='file':
        filename= urllib.url2pathname(uri[2])

    # Code must be line-terminated or some indented constructs won't compile.
    #
    if code[-1:]!='\n':
      code= code+'\n'

    # Prefix the code with future-feature imports if there are any needed.
    #
    if len(self.features)>0:
      features= string.join(self.features, ', ')
      code= 'from __future__ import %s\n%s' % (features, code)
      if lineNumber is not None:
         lineNumber= lineNumber-1

    # Compile into a code object. This can fail if the code is badly-formed
    # (SyntaxError, UnicodeError) - fiddle a SyntaxError's line number to
    # match the context.
    #
    try:
      if convertExecs:
        code= str(code)
      cobj= compile(code, filename or '<template>', 'exec')
    except NonErrors:
      raise
    except:
      ei= sys.exc_info()
      if isinstance(ei[1], SyntaxError) and lineNumber is not None:
        ei[1].lineno= ei[1].lineno+lineNumber
      raise PXTLRuntimeError(context, ei)

    # Fiddle the line numbers of the compiled code object and attempt to
    # execute it. Wrap all errors except for ones we raised in the first place
    # (which are passed through).
    #
    if lineNumber is not None:
      cobj= renumberCode(cobj,
        xrange(lineNumber-1, lineNumber+string.count(code, '\n')+1)
      )
    try:
      exec cobj in self.getGlobals(), self.getLocals()
    except NonErrors+(PXTLError,):
      raise
    except:
      ei= sys.exc_info()
      raise PXTLRuntimeError(context, (ei[0], ei[1], ei[2]))


  def pxexec(self, code, context):
    """ Execute single line of Python code in scope.
    """
    code= string.lstrip(code)
    if code=='':
      return
    self._pxcode(code, context)

  def pxeval(self, expr, context):
    """ Evaluate Python expression in scope.
    """
    # Can't use 'eval' itself as _pxcode may need to add future-feature import
    # lines etc. Have the code write to a temporary variable in local scope
    # then extract the value from scope itself.
    #
    self._pxcode('__pxtl_eval= (%s)' % expr, context)
    scope= self.getLocals()
    return scope['__pxtl_eval']

  def pxassign(self, var, value, context):
    """ Assign something to a Python l-value in scope.
    """
    # Again, have to use a temporary variable to pass in to code - can't write
    # directly to scope as l-values can be (much) more complicated than just a
    # variable name.
    #
    scope= self.getLocals()
    scope['__pxtl_assign']= value
    self._pxcode('(%s)= __pxtl_assign' % string.lstrip(var), context)

  def pxblock(self, code, context):
    """ Execute multiple lines of code indented with an arbitrary amount of
        whitespace.
    """
    # Find first non-empty, non-comment line and see if it is indented. If
    # there is no such line we have no code, return without doing anything.
    #
    for line in string.split(code, '\n'):
      stripped= string.lstrip(line)
      if stripped!='' and stripped[0]!='#':
        indented= len(stripped)!=len(line)
        break
    else:
      return

    # If the block is indented, put it in a dummy construct that requires
    # indentation. If not, add a single dummy-line just to make the line
    # numbering tie up (because in a "<?px :" construct, line 1 is actually
    # the line after the colon, not the start of the PI).
    #
    if indented:
      self._pxcode('if 1:\n%s' % code, context)
    else:
      self._pxcode('pass\n%s' % code, context)


def removePXTLNamespaces(node):
  """ Once all PXTL elements and attributes have been processed away, remove
      namespace declarations that refer to PXTL.
  """
  if node.attributes is not None:
    for attr in attrList(node.attributes):
      if attr.namespaceURI==NSNS:
        if attr.value==PXNS:
          node.removeAttributeNode(attr)
  for child in list(node.childNodes):
    if child.nodeType==child.ELEMENT_NODE:
      removePXTLNamespaces(child)
