""" Utility functions used by PXTL implementations.
"""

__all__= [
  'attrList', 'removeReparent', 'setDoctype', 'getNamespaces',
  'DictAsInstance', 'encodeText', 'decodeName', 'parsePseudoPIs',
  'ConditionalContext', 'checkFeatures', 'haveNestedScopes',
  'AbstractPXTLObject', 'mapPath', 'determineMethod', 'writeTypeHeader',
  'importRead', 'getLogicalLines', 'getIndentSize', 'renumberCode',
  'evalStatic', 'lastItem'
]

import types, sys, string, re, urllib, pxtl, new
from pxtl.constants import *
from pxtl.exceptions import *


# DOM utility functions
#
def attrList(nodes):
  """ Turn a live DOM NamedNodeMap into a static Python list, which can be
      used for potentially destructive iteration.
  """
  l= []
  for i in range(nodes.length):
    l.append(nodes.item(i))
  return l


def removeReparent(node):
  """ Replace a DOM node with its children, making sure all namespaces are
      the same on the child by copying declarations where necessary.
  """
  for i in range(node.attributes.length):
    attribute= node.attributes.item(i)
    if attribute.namespaceURI==NSNS and attribute.value!=PXNS:

      # Ignore declarations that are already present on parent
      #
      if node.parentNode is not None:
        if node.parentNode.nodeType!=node.DOCUMENT_NODE:
          if attribute.prefix is None:
            prefix= None
          else:
            prefix= attribute.localName
          value= attribute.value
          if value=='':
            value= None
          if node.parentNode.lookupNamespaceURI(prefix)==value:
            continue

      # Ignore declarations that are overridden (declared on child). Copy the
      # rest to each child.
      #
      for child in list(node.childNodes):
        if child.nodeType==child.ELEMENT_NODE:
          if not child.hasAttributeNS(NSNS, attribute.localName):
            child.setAttributeNS(NSNS, attribute.name, attribute.value)

  # Replace node with a DocumentFragment of all its children.
  #
  parent= node.parentNode
  frag= node.ownerDocument.createDocumentFragment()
  for child in list(node.childNodes):
    if parent.nodeType!=node.DOCUMENT_NODE or child.nodeType!=node.TEXT_NODE:
      frag.appendChild(child)
  parent.replaceChild(frag, node)


def setDoctype(document, doctype):
  """ Try to set the doctype of a document to the specified object. This can't
      be done directly in DOM Level 2, and in Level 3 it's still not required,
      so direct replacement may not be possible with implementations other
      than pxdom. In this case, clone the document's content into a new
      document with the correct doctype.
  """
  current= document.doctype
  try:
    if current is None:
      if doctype is None:
        return document
      else:
        document.insertBefore(doctype, document.firstChild)
    else:
      if doctype is None:
        document.removeChild(current)
      else:
        document.replaceChild(doctype, current)
    return document
  except NonErrors:
    raise
  except Exception:

    # Failed in some way (why can't tell what, since DOMExceptions are not
    # bound to anything in the Python bindings - the exception could be
    # anything). Try the copying method instead.
    #
    imp= document.implementation
    newDocument= imp.createDocument(PXNS, 'px:none', doctype)
    newDocument.xmlVersion= document.xmlVersion
    refEl= newDocument.documentElement
    for child in list(document.childNodes):
      if child.nodeType==child.ELEMENT_NODE:
        newDocument.replaceChild(
          newDocument.importNode(child, True), newDocument.documentElement
        )
        refEl= None
      elif child.nodeType!=child.DOCUMENT_TYPE_NODE:
        newDocument.insertBefore(newDocument.importNode(child, True), refEl)
    return newDocument


def getNamespaces(node):
  """ Get a mapping {prefix: uri} of all non-built-in namespaces active on an
      element.
  """
  nss= {}
  while node.nodeType==node.ELEMENT_NODE:
    for attr in attrList(node.attributes):
      if attr.namespaceURI==NSNS:
        prefix= attr.localName
        if attr.localName=='xmlns':
          prefix= None
        if not nss.has_key(prefix):
          nss[prefix]= attr.value
    node= node.parentNode
  return nss


# General-purpose utility functions.
#
class DictAsInstance:
  def __init__(self, d):
    self.__dict__= d

_encode_upar= re.compile('[^a-zA-Z0-9\\-\\._]')
_encode_jstr= re.compile('[\\\\\'"&<>\\-]')
_encode_cstr= _encode_jstr
_encode_cstr_u= re.compile('\\\\u....')
_encode_name= re.compile('[^a-zA-Z0-9\\-\\.]')
_decode_name= re.compile('_..')

_hexdigits= '0123456789ABCDEF0123456789abcdef'

def encodeText(text, coding= 'text'):
  """ Return text in an escaped form (for URL parameters, JS literals etc.).
  """
  if coding=='text':
    return text
  elif coding=='upar':
    if type(text)==UnicodeType:
      text= text.encode('utf-8')
    return _encode_upar.sub(lambda m: '%%%02X' % ord(m.group(0)), text)
  elif coding=='jstr':
    text= _encode_jstr.sub(lambda m: '\\x%02X' % ord(m.group(0)), text)
    if type(text)==UnicodeType:
      text= text.encode('unicode-escape')
    return text
  elif coding=='cstr':
    text= _encode_cstr.sub(lambda m: '\\%02X ' % ord(m.group(0)), text)
    if type(text)==UnicodeType:
      text= text.encode('unicode-escape')
      text= _encode_cstr_u.sub(lambda m: '\\%s ' % m.group(0)[2:], text)
    return text
  elif coding=='name':
    if text=='':
      return '___'
    if type(text)==UnicodeType:
      text= text.encode('utf-8')
    text= _encode_name.sub(lambda m: '_%02X' % ord(m.group(0)), text)
    if text[0] in '0123456789-.':
      text= ('_%02X' % ord(text[0]))+text[1:]
    return text
  else:
    raise ValueError ("unknown coding '%s'" % coding)

def _decodeName_sub(m):
  c= m.group(0)[1:]
  if c[0] in _hexdigits and c[1] in _hexdigits:
    return chr(eval('0x'+c))
  return ''
def decodeName(text):
  text= _decode_name.sub(_decodeName_sub, text)
  if UnicodeType is None:
    return text
  return unicode(text, 'utf-8')


_ws= [' ', '\n', '\t']
def parsePseudoPIs(v, node= None):
  """ Break an attribute value up into text and pseudo-PI sections
      (target, data).
  """
  parts= []
  ix= 0
  while True:
    ox= string.find(v, '{?', ix)
    if ox==-1:
      if ix<len(v):
        parts.append(v[ix:])
      break
    if ox>ix:
      parts.append(v[ix:ox])
    cx= string.find(v, '?}', ox+1)
    if cx==-1:
      raise PXTLPseudoPIOpenError(node, ox)
    if cx==ox+1:
      parts.append('{?')
    else:
      wx= ox+2
      while wx<cx:
        if v[wx] in _ws:
          break
        wx= wx+1
      vx= wx
      while vx<cx:
        if v[vx] not in _ws:
          break
        vx= vx+1
      parts.append((v[ox+2:wx], v[vx:cx]))
    ix= cx+2
  return parts


def checkFeatures(future):
  """ Convert a future attribute node into a list of optional extra features.
      Raise an exception for any features that cannot be used in the current
      version of Python.
  """
  if future==None:
    return []
  features= map(string.strip, string.split(future.value, ','))
  for feature in features[:]:

    # If the feature is not implemented at all in the current version of
    # Python, complain. If the feature is on by default anyway, don't bother
    # including it in the import list.
    #
    if hasattr(future__mod, feature):
      f= getattr(future__mod, feature)
      if sys.version_info>=f.getOptionalRelease():
        if sys.version_info>=f.getMandatoryRelease():
          features.remove(feature)
        continue
    raise PXTLFutureFeatureError(future, feature)
  return features

def haveNestedScopes(features):
  """ Determine whether PXTL should provide nested scopes to subtemplate
      functions. Check the supplied features string and the current Python
      version's default features.
  """
  if 'nested_scopes' in features:
    return True
  if hasattr(future__mod, 'nested_scopes'):
    if sys.version_info>=future__mod.nested_scopes.getMandatoryRelease():
      return True
  return False

class ConditionalContext:
  """ Keeps track of previous clauses of an if-orif-anif-elif-else construct
      and calculates the success of each element. Used by conditional elements
      and conditional attributes.
  """
  def reset(self):
    """ Start a new conditional context, forgetting any previous clauses. This
        also serves as the constructor.
    """
    self.old_cond= None
    self.old_testing= None
    self.old_success= None
  __init__= reset

  def getTesting(self, cond, node):
    """ Given the new clause 'cond' and knowledge of previous clauses, decide
        whether the clause's test condition needs to be evaluated to determine
        its success. If not laziness kicks in and we can avoid doing the test.
    """
    if cond=='if':
      return True
    if self.old_cond not in PRECONDITIONALS:
      raise PXTLConditionalOrderError(node, self.old_cond)
    if cond=='else':
      return False
    if cond=='orif':
      return not self.old_success
    if cond=='anif':
      return self.old_success
    if cond=='elif':
      if self.old_cond=='elif':
        return self.old_testing and not self.old_success
      else:
        return not self.old_success
    raise ValueError('Unknown conditional %s' % cond)

  def getSuccess(self, cond, test):
    """ Work out whether a clause is successful, given the truth of its
        evaluated test condition. If the condition was not evaluated (due to
        getTesting having returned False), 'test' should be None instead of a
        boolean. Store the clause info as it may affect a following clause.
    """
    if cond=='else':
      if self.old_cond=='elif':
        success= self.old_testing and not self.old_success
      else:
        success= not self.old_success
    elif cond=='orif':
      success= test in (None, True)
    else:
      success= test==True
    self.old_cond= cond
    self.old_testing= test is not None
    self.old_success= success
    return success

  def copy(self):
    """ Make a copy of a ConditionalContext which can be further updated
        without affecting this one (for nesting).
    """
    clone= ConditionalContext()
    clone.old_cond= self.old_cond
    clone.old_testing= self.old_testing
    clone.old_success= self.old_success
    return clone


class AbstractPXTLObject:
  """ Object passed into all template global scopes for output-writing and
      constant-holding purposes.
  """
  version= 1,1
  pxtl= pxtl

  TEXT=    ('text/plain',             'text', None, None)
  CSS=     ('text/css',               'text', None, None)
  JS=      ('text/javascript',        'text', None, None)
  XML=     ('text/xml',               'xml',  None, None)
  TAGSOUP= ('text/html',              'html',
              '-//W3C//DTD HTML 4.01 Transitional//EN', None)
  HTML2=   ('text/html',              'html',
              '-//IETF//DTD HTML 2.0//EN', None)
  HTML3=   ('text/html',              'html',
              '-//W3C//DTD HTML 3.2 Final//EN', None)
  HTML4S=  ('text/html',              'html',
              '-//W3C//DTD HTML 4.01//EN',
              'http://www.w3.org/TR/html4/strict.dtd')
  HTML4T=  ('text/html',              'html',
              '-//W3C//DTD HTML 4.01 Transitional//EN',
              'http://www.w3.org/TR/html4/loose.dtd')
  HTML4F=  ('text/html',              'html',
              '-//W3C//DTD HTML 4.01 Frameset//EN',
              'http://www.w3.org/TR/html4/frameset.dtd')
  XHTML1S= ('text/html',              'xhtml',
              '-//W3C//DTD XHTML 1.0 Strict//EN',
              'http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd')
  XHTML1T= ('text/html',              'xhtml',
              '-//W3C//DTD XHTML 1.0 Transitional//EN',
              'http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd')
  XHTML1F= ('text/html',              'xhtml',
              '-//W3C//DTD XHTML 1.0 Frameset//EN',
              'http://www.w3.org/TR/xhtml1/DTD/xhtml1-frameset.dtd')
  XHTML11= ('application/xhtml+xml',   'xhtml',
              '-//W3C//DTD XHTML 1.1//EN',
              'http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd')
  XHTMLB1= ('application/xhtml+xml',   'xml',
              '-//W3C//DTD XHTML Basic 1.0//EN',
              'http://www.w3.org/TR/xhtml-basic/xhtml-basic10.dtd')
  XHTMLM1= ('application/xhtml+xml',   'xml',
              '-//WAPFORUM//DTD XHTML Mobile 1.0//EN',
              'http://www.wapforum.org/DTD/xhtml-mobile10.dtd')
  IXHTML1= ('application/xhtml+xml',   'xml',
              '-//i-mode group (ja)//DTD XHTML i-XHTML (Locale/Ver.=ja/1.0) 1.0//EN',
              'i-xhtml_4ja_10.dtd')
  IXHTML11=('application/xhtml+xml',   'xml',
              '-//i-mode group (ja)//DTD XHTML i-XHTML (Locale/Ver.=ja/1.1) 1.0//EN',
              'i-xhtml_4ja_10.dtd')
  WML11=   ('application/xhtml+xml',   'xml',
              '-//WAPFORUM//DTD WML 1.1//EN',
              'http://www.wapforum.org/DTD/wml_1.1.xml')
  WML12=   ('application/xhtml+xml',   'xml',
              '-//WAPFORUM//DTD WML 1.2//EN',
              'http://www.wapforum.org/DTD/wml12.dtd')
  WML13=   ('application/xhtml+xml',   'xml',
              '-//WAPFORUM//DTD WML 1.3//EN',
              'http://www.wapforum.org/DTD/wml13.dtd')
  WML20=   ('application/xhtml+xml',   'xml',
              '-//WAPFORUM//DTD WML 2.0//EN',
              'http://www.wapforum.org/dtd/wml20.dtd')
  MATH1=   ('application/mathml+xml', 'xml',
              None, 'http://www.w3.org/Math/DTD/mathml1/mathml.dtd')
  MATH2=   ('application/mathml+xml', 'xml',
              '-//W3C//DTD MathML 2.0//EN',
              'http://www.w3.org/TR/MathML2/dtd/mathml2.dtd')
  SVG1=    ('image/svg+xml',          'xml',
              '-//W3C//DTD SVG 1.0//EN',
              'http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd')
  SVG11=   ('image/svg+xml',          'xml',
              '-//W3C//DTD SVG 1.1//EN',
              'http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd')
  SVG11B=  ('image/svg+xml',          'xml',
              '-//W3C//DTD SVG 1.1 Basic//EN',
              'http://www.w3.org/Graphics/SVG/1.1/DTD/svg11-basic.dtd')
  SVG11T=  ('image/svg+xml',          'xml',
              '-//W3C//DTD SVG 1.1 Tiny//EN',
              'http://www.w3.org/Graphics/SVG/1.1/DTD/svg11-tiny.dtd')
  XHMS=    ('application/xhtml+xml',   'xml',
              '-//W3C//DTD XHTML 1.1 plus MathML 2.0 plus SVG 1.1//EN', 
              'http://www.w3.org/2002/04/xhtml-math-svg/xhtml-math-svg.dtd')


def mapPath(path, mappings):
  """ Send a given path through the mapping process to work out where to store
      bytecode file. Look in the mapping keys for the nearest ancestor path.
      If there is one, replace that ancestor-part of the path with the
      corresponding mapping value. If not, just return the original path.
  """
  # Normalise input paths for easier comparison
  #
  norm= lambda x: os.path.normcase(os.path.abspath(x))
  path= norm(path)
  maps= {}
  for key, value in mappings.items():
    maps[norm(key)]= norm(value)

  anc= path
  desc= None
  while True:

    # See if the current path matches a mapping
    #
    if maps.has_key(anc):
      path= maps[anc]
      if desc is not None:
        path= os.path.join(path, desc)
      return path

    # Step up a parent and try again, until we hit the root
    #
    anc, cur= os.path.split(anc)
    if cur=='':
      return path
    if desc is None:
      desc= cur
    else:
      desc= os.path.join(cur, desc)


def determineMethod(doctuple, namespaceURI, localName):
  """ Choose which output method to use given an evaluated doctype attribute
      and the name of the root node.
  """
  method= doctuple[1]
  if method is not None:
    return method
  if namespaceURI==HTNS or (namespaceURI is None and localName=='html'):
    return 'xhtml'
  else:
    return 'xml'
  
  
def writeTypeHeader(writer, mimetype, method, encoding):
  """ Send an HTTP Content-Type response header best matching the document.
  """
  if mimetype is None:
    if method in ('html', 'xhtml'):
      mimetype= 'text/html'
    elif method=='xml':
      mimetype= 'text/xml'
    else:
      mimetype= 'text/plain'
  if encoding is not None and string.lower(mimetype[:5])=='text/':
    writer.write(
      'Content-Type: %s; charset=%s\r\n\r\n' % (str(mimetype), str(encoding))
    )
  else:
    writer.write('Content-Type: %s\r\n\r\n' % str(mimetype))


def importRead(src, node):
  """ Trivial method used by an optimised run-time program or the reference
      implementation. Try to open a stream using urllib, raise a particular
      exception if it fails.
  """
  try:
    return urllib.urlopen(src)
  except IOError:
    raise PXTLImportSrcError(node, src)


def getLogicalLines(block):
  """ Split a multi-line code block string into Python logical lines, removing
      any common indentation.
  """
  # (Non-logical lines include empty lines, comment lines, triple-quoted
  # strings, backslash-continuations and bracketed expressions.)
  #
  # Finding out which lines are logical lines in Python code is not normally
  # possible. We cheat by compiling the code over and over again, each time
  # changing the spacing of one line. If that line is a logical line, changing
  # the spacing will cause an IndentationError. If no error occurs we know
  # that physical line is not a logical line.
  #
  # This can be a bit slow, so it is only used in the optimised implementation
  # at compile-time. The reference implementation can get away with a
  # simpler hack than redentation as its code block indent levels are flat.

  # Get the common indentation by looking at the indentation of the first
  # non-empty, non-comment line.
  #
  lines= string.split(block, '\n')
  for line in lines:
    undent= getIndentSize(line)
    if undent is not None:
      break
  else:
    return [block]

  # Make an indented copy of the lines so they can definitely be included in
  # an if-block. (Use eight spaces so that there can be no interaction with
  # any following tabs). Check the results are compilable before we start
  # messing with the indentation.
  #
  execblock= 'if 1:\n        %sdoNothing\n%%s\n' % (' '*undent)
  testlines= map(lambda x: '        '+x, lines)
  compile(execblock % string.join(testlines, '\n'), '<test>', 'exec')

  # Iterate over lines of code, screwing up a single line's indentation at a
  # time by replacing any indentation it has with a single space character;
  # this can never match a consistent indent level (as we have indented the
  # rest by 8 spaces).
  #
  logicals= []
  sink= StreamSink()
  for ix in range(len(testlines)):
    line= testlines[ix]
    testlines[ix]= ' '+string.lstrip(line)

    # Try to compile the result. If there's an error, add the current line to
    # the list of logical lines, first removing any common indentation.
    # Aggravatingly, Python 1.6 and earlier send the message 'inconsistent
    # dedent' to stderr as well as throwing the exception; we don't want this,
    # so temporarily redirect stderr to oblivion.
    #
    try:
      if sys.hexversion<0x02000000:
        oldStderr= sys.stderr
        sys.stderr= sink
      try:
        compile(execblock % string.join(testlines, '\n'), '<test>', 'exec')
      finally:
        if sys.hexversion<0x02000000:
          sys.stderr= oldStderr
    except SyntaxError:
      logicals.append(
        ' '*(getIndentSize(lines[ix])-undent) + string.lstrip(lines[ix])
      )

    # If no error, add the physical line to the previous logical line (if any)
    #
    else:
      if logicals==[]:
        logicals.append(lines[ix])
      else:
        logicals[-1]= logicals[-1]+'\n'+lines[ix]

    # Restore the indentation and move onto the next physical line.
    #
    testlines[ix]= line
  return logicals


def getIndentSize(line):
  """ Return the number of spaces a line's indent is equivalent to. If the
      line is empty or a comment line, the indentation is irrelevant and None
      is returned.
  """
  stripped= string.lstrip(line)
  if stripped=='' or stripped[:1]=='#':
    return None
  indent= line[:len(line)-len(stripped)]
  if '\t' not in indent:
    return len(indent)
  size= 0
  for c in indent:
    if c=='\t':
      size= size+TABSIZE-(size%TABSIZE)
    else:
      size= size+1
  return size
TABSIZE= 8 # currently defined by Python


class StreamSink:
  """ A /dev/null stream as a class.
  """
  def write(self, s):
    pass
  def read(self):
    return ''


def evalStatic(expr):
  """ Compile and evaluate an expression if it is completely static, that is
      it does not depend on or alter anything from a scope. This means any
      expression containing only literals. Expressions that cause errors are
      never static. Return a tuple of is-static-flag and static value.
  """
  # Decide that an expression is not static if it contains non-literal names.
  # For the benefit of the common case, pretend that None, False and True are
  # literals, not values. (As they probably will be in future-Python.)
  #
  try:
    code= compile(string.strip(expr), '<evalStatic>', 'eval')
  except SyntaxError:
    return (False, None)
  for name in code.co_names:
    if name not in ('None', 'False', 'True'):
      return (False, None)
  try:
    return (True, eval(code, STATICSCOPE))
  except NonErrors:
    raise
  except Exception:
    return (False, None)
STATICSCOPE= {'None': None, 'False': False, 'True': True}


def renumberCode(code, linetrans):
  """ Pass the line numbers set in a Python code object (and any code object
      it contains) through a translation list so they match the positions in
      the original PXTL template instead of the temporary code text they were
      compiled with. This allows execution frames (including exception
      tracebacks) to give useful line numbers and content automatically.
  """
  lnotab= StringIO()

  progline= code.co_firstlineno
  if len(linetrans)<=progline:
    progline= len(linetrans)-1
  firstlineno= linetrans[progline]

  srcline= firstlineno
  ix= 0
  while ix<len(code.co_lnotab)-1:
    bytes, lines= ord(code.co_lnotab[ix]), ord(code.co_lnotab[ix+1])
    ix= ix+2

    progline= progline+lines
    if len(linetrans)<=progline:
      progline= len(linetrans)-1
    increment= linetrans[progline]-srcline
    if increment<0:
      increment= 0
    srcline= linetrans[progline]

    while increment>=256:
      lnotab.write(chr(bytes))
      lnotab.write('\xFF')
      increment= increment-255
      bytes= 0
    lnotab.write(chr(bytes))
    lnotab.write(chr(increment))

  # Check for other constant code blocks (def statements) compiled into this
  # one and fix them up too.
  #
  consts= []
  for const in code.co_consts:
    if type(const)==type(code):
      const= renumberCode(const, linetrans)
    consts.append(const)

  # Rewrap the code object with altered line numbers and constants. Remember
  # to include cellvars and freevars if they are available (Python 2.1+); this
  # will need to be extended if the code object's properties are extended in a
  # future version of Python.
  #
  args= [
    code.co_argcount, code.co_nlocals, code.co_stacksize, code.co_flags,
    code.co_code, tuple(consts), code.co_names, code.co_varnames,
    code.co_filename, code.co_name, firstlineno, lnotab.getvalue()
  ]
  if hasattr(code, 'co_freevars'):
    args.append(code.co_freevars)
  if hasattr(code, 'co_cellvars'):
    args.append(code.co_cellvars)
  return apply(new.code, args)


# Utility function to return the last non-None item in a list, used for run-
# time whitespace handling.
#
def lastItem(l):
  for i in range(len(l)):
    if l[len(l)-i-1] is not None:
      return l[len(l)-i-1]
  return None
