""" Distutils script. Use 'python setup.py install' to install pxtl.
"""

import os, sys, string, stat
from distutils.core import setup
from distutils.command.build_py import build_py
from distutils.command.install_lib import install_lib


# The pxtl package requires the system template '__error__.px' to be included
# in its package directory. Normally distutils does not allow non-module files
# to be included in a package directory, so override the build_py command to
# copy .px files as well as .py files to the build directory.
#
class build_px(build_py):
  def build_packages(self):
    build_py.build_packages(self)
    for package in self.packages:
      package_dir= self.get_package_dir(package)
      for template in os.listdir(package_dir):
        if os.path.splitext(template)[1]=='.px':
          infile= os.path.join(package_dir, template)
          outpath= [self.build_lib]+string.split(package, '.')+[template]
          self.copy_file(infile,apply(os.path.join, outpath),preserve_mode=0)


# The '__error__.px' template can be bytecode-compiled along with the normal
# modules, to improve performance when used with the optimised implementation.
# Extend the install_lib command to invoke pxtl.optimised's bytecode compiler
# for .px files in the build directory.
#
class install_px(install_lib):
  def run(self):
    install_lib.run(self)
    if self.compile:
      from pxtl.optimised import compileFile
      for path in getPaths(self.build_dir, self.install_dir):
        if os.path.splitext(path)[1]=='.px':
          pathc= path+'c'
          leafc= os.path.split(pathc)[1]

          compile= self.force
          if not compile:
            compile= 1
            if os.path.exists(pathc):
              mt, mtc= map(lambda f: os.stat(f)[stat.ST_MTIME], (path, pathc))
              compile= mt>mtc
          if compile:
            print "byte-compiling %s to %s" % (path, leafc)
            compileFile(path, pathc)

# Utility function for install_px. Get pathnames of all descendant files in a
# directory, mapped to be relative to another directory. Normally distutils
# does this with copy_tree but that does a load of unwanted logging and stuff.
#
def getPaths(parent, base):
  paths= []
  for child in os.listdir(parent):
    path= os.path.join(parent, child)
    mapped= os.path.join(base, child)
    if os.path.isdir(path):
      paths.extend(getPaths(path, mapped))
    else:
      paths.append(mapped)
  return paths


# Hack old distutils versions so they don't complain about the new PyPI
# arguments.
#
if sys.version < '2.2.3':
  from distutils.dist import DistributionMetadata
  DistributionMetadata.classifiers = None
  DistributionMetadata.download_url = None


# Invoke distutils, using our own extended command classes.
#
setup(
  packages= ['pxtl'],
  cmdclass= {'build_py': build_px, 'install_lib': install_px},

  name= 'pxtl',
  version= '1.5',
  license= 'http://www.opensource.org/licenses/bsd-license.php',

  author= 'Andrew Clover',
  author_email= 'and@doxdesk.com',
  url= 'http://www.doxdesk.com/software/py/pxtl.html',
  download_url= 'http://www.doxdesk.com/file/software/py/pxtl.zip',

  description= 'Python XML Templating Language',
  long_description=
    'The pxtl package provides two complementary implementations of the '
    'Python XML Templating Language, an XML document type for transforming '
    'XML, HTML and other textual languages. PXTL is well-suited to rapid '
    'development both of full-scale dynamic web applications and '
    'general-purpose XML transformations.',
  classifiers= [
    'Development Status :: 5 - Production/Stable',
    'Environment :: Web Environment',
    'Intended Audience :: Developers',
    'License :: DFSG approved',
    'License :: OSI Approved :: BSD License',
    'Operating System :: OS Independent',
    'Topic :: Text Processing :: Markup :: XML',
    'Topic :: Internet :: WWW/HTTP :: Dynamic Content'
  ]
)
